<?php

declare(strict_types=1);

namespace Atlas\AuditBundle\Twig\Component;

use Atlas\AuditBundle\Repository\SystemRepository;
use Modules\Paginator\Service\Paginator;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveProp;

/**
 * Live audit viewer: renders audits + paginates without full page reload.
 */
#[AsLiveComponent(name: 'audit_viewer', template: '@Audit/components/audit_viewer.html.twig')]
final class AuditViewerComponent
{
    /** Audit table_name to filter by. */
    #[LiveProp]
    public string $table_name;

    /** Subject identifier to filter by. */
    #[LiveProp]
    public int $identifier;

    /** Items per page. */
    #[LiveProp]
    public int $per_page = 50;

    /** Current audit page (1-based). */
    #[LiveProp(writable: true)]
    public int $page = 1;

    /**
     * @var array|null
     */
    private ?array $audits_page_cache = null;

    public function __construct(
        private readonly SystemRepository $audits,
        private readonly Paginator $paginator,
    ) {}

    /**
     * Get the current paginated audits page.
     *
     * @return array
     */
    public function getAuditsPage(): array
    {
        if ($this->audits_page_cache !== null) {
            return $this->audits_page_cache;
        }

        if ($this->page < 1) {
            $this->page = 1;
        }

        $query = $this->audits->findForSubjectQueryBuilder($this->table_name, $this->identifier);

        return $this->audits_page_cache = $this->paginator->paginate(
            $query,
            $this->page,
            max(1, $this->per_page),
        );
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function getRows(): array
    {
        $page = $this->getAuditsPage();

        $rows = [];

        foreach ($page['items'] as $audit) {
            $rows[] = [
                'action' => $audit->action,
                'reason' => $audit->reason,
                'user' => $audit->user,
                'timestamp' => $audit->timestamp,
                'original' => $audit->original_values_array,
                'changed' => $audit->changed_values_array,
            ];
        }

        return $rows;
    }

    #[LiveAction]
    public function gotoPage(#[LiveArg] int $page): void
    {
        $this->page = max(1, $page);
        $this->audits_page_cache = null;
    }

    #[LiveAction]
    public function nextPage(): void
    {
        $this->page = max(1, $this->page + 1);
        $this->audits_page_cache = null;
    }

    #[LiveAction]
    public function prevPage(): void
    {
        $this->page = max(1, $this->page - 1);
        $this->audits_page_cache = null;
    }
}