<?php

declare(strict_types=1);

namespace Atlas\AuditBundle\Listener;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Entity\System;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\Bundle\DoctrineBundle\Attribute\AsDoctrineListener;
use Doctrine\ORM\Event\PostFlushEventArgs;
use Doctrine\ORM\Event\PostPersistEventArgs;
use Doctrine\ORM\Event\PostUpdateEventArgs;
use Doctrine\ORM\Event\PreRemoveEventArgs;
use Doctrine\ORM\Events;
use Doctrine\ORM\Mapping\JoinColumn;
use Doctrine\Persistence\ObjectManager;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use ReflectionException;
use ReflectionProperty;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

#[AsDoctrineListener(event: Events::postPersist)]
#[AsDoctrineListener(event: Events::postUpdate)]
#[AsDoctrineListener(event: Events::preRemove)]
#[AsDoctrineListener(event: Events::postFlush)]
final class LoggerListener
{
    /** @var System[] */
    private array $logs = [];

    //hard to prevent recursive flushing
    private bool $is_flushing = false;

    //save reduce reflection calls
    private array $reflection_cache = [];

    private array $connections = [];

    public function __construct(
        private readonly LoggerInterface $logger,
        private readonly ManagerRegistry $registry,
        #[Autowire(param: 'shared.log.connections')] string $connections
    )
    {
        $connections = trim($connections);

        if($connections !== '') {
            $this->connections = array_values(array_filter(
                array_map('trim', explode(',', $connections)),
                static fn (string $c): bool => $c !== ''
            ));
        }
    }

    /**
     * @param PostPersistEventArgs $event
     * @return void
     */
    public function postPersist(PostPersistEventArgs $event): void
    {
        $this->handleEvent($event->getObjectManager(), $event->getObject(), AuditActionType::INSERT);
    }

    /**
     * @param PostUpdateEventArgs $event
     * @return void
     */
    public function postUpdate(PostUpdateEventArgs $event): void
    {
        $this->handleEvent($event->getObjectManager(), $event->getObject(), AuditActionType::UPDATE);
    }

    /**
     * @param PreRemoveEventArgs $event
     * @return void
     */
    public function preRemove(PreRemoveEventArgs $event): void
    {
        $this->handleEvent($event->getObjectManager(), $event->getObject(), AuditActionType::DELETE);
    }

    /**
     * @param PostFlushEventArgs $event
     * @return void
     */
    public function postFlush(PostFlushEventArgs $event): void
    {
        if ($this->is_flushing || empty($this->logs)) {
            return;
        }

        $this->is_flushing = true;
        $em = $event->getObjectManager();

        foreach ($this->logs as $log) {
            $em->persist($log);
        }

        $this->logs = [];
        $em->flush();
        $this->is_flushing = false;
    }

    /**
     * @param ObjectManager $em
     * @param object $object
     * @param AuditActionType $action
     * @return void
     */
    private function handleEvent(ObjectManager $em, object $object, AuditActionType $action): void
    {

        if($this->connections !== []) {

            $allowed = false;

            foreach($this->connections as $connection) {
                $expected = $this->registry->getManager($connection);

                if($expected === $em) {
                    $allowed = true;
                    break;
                }
            }
            if(! $allowed) return;
        }

        $ref =  $this->getReflectionClass($object);

        //check class level not logged
        foreach ($ref->getAttributes() as $attr) {
            if (is_a($attr->getName(), NotLogged::class, true)) {
                return;
            }
        }

        $log = $this->log($em, $object, $action);
        if ($log) {
            $this->logs[] = $log;
        }
    }

    /**
     * @param ObjectManager $em
     * @param object $object
     * @param AuditActionType $action
     * @return System|null
     */
    private function log(ObjectManager $em, object $object, AuditActionType $action): ?System
    {
        $uow = $em->getUnitOfWork();
        $meta = $em->getClassMetadata(get_class($object));
        $idents = $meta->getIdentifierValues($object);
        $entity = $uow->getOriginalEntityData($object);
        $changes = $uow->getEntityChangeSet($object);

        $id = $idents['id'] ?? null;

        if ($id === null) {
            $id = $entity['id'] ?? null;
        }

        if ($id === null) {
            $this->logger->warning("No identifier was specified.", [
                'class' => get_class($object),
                'action' => $action
            ]);
            return null; // No valid ID, skip logging
        }

        $original = $updated = [];

        try {
            if ($action === AuditActionType::DELETE) {
                foreach ($entity as $key => $value) {
                    $info = $this->shouldIncludeField($object, $meta, $key);
                    if (!$info) { continue; }

                    $this->setOriginalUpdated(
                        $original,
                        $updated,
                        $info['mapKey'],
                        $info['isJoin'],
                        $value,   // old
                        null,     // new (unused for delete)
                        AuditActionType::DELETE
                    );
                }
            } else {
                // For INSERT/UPDATE, log changed values
                foreach ($changes as $key => $value) {
                    $info = $this->shouldIncludeField($object, $meta, $key);
                    if (!$info) { continue; }

                    // $value = [old, new]
                    $this->setOriginalUpdated(
                        $original,
                        $updated,
                        $info['mapKey'],
                        $info['isJoin'],
                        $value[0],
                        $value[1],
                        $action
                    );
                }
            }
        } catch (ReflectionException $e) {
            $this->logger->critical('Reflection failed on logger: ' . $e->getMessage(), [
                    'exception_class' => get_class($e),
                    'property' => $key ?? 'unknown',
                    'class' => get_class($object),
                    'trace' => $e->getTraceAsString(),
                    'action' => $action
                ]
            );
        }

        $timestamp = $this->resolveTimestamp($object, $action);
        $actionBy  = $this->resolveActor($object, $action);

        //reason might not change so if not try and get it from the change or if not get it from the original
        if (property_exists($object, 'reason')) {
            $reason = $object->reason;
        } else {
            $reason = 'No reason recorded';
        }

        return new System(
            $id,
            $meta->getTableName(),
            $action->value,
            $reason,
            $actionBy,
            originalValues: $original,
            changedValues: $updated,
            timestamp: $timestamp
        );
    }

    /**
     * @param object $object
     * @return ReflectionClass
     */
    private function getReflectionClass(object $object): ReflectionClass
    {
        $class = get_class($object);
        if (!isset($this->reflection_cache[$class]['_class'])) {
            $this->reflection_cache[$class]['_class'] = new ReflectionClass($object);
        }
        return $this->reflection_cache[$class]['_class'];
    }

    /**
     * @param object $object
     * @param string $propertyName
     * @return ReflectionProperty
     * @throws ReflectionException
     */
    private function getReflectionProperty(object $object, string $propertyName): ReflectionProperty
    {
        $class = get_class($object);

        if (!isset($this->reflection_cache[$class][$propertyName])) {
            $refClass = $this->getReflectionClass($object);
            $this->reflection_cache[$class][$propertyName] = $refClass->getProperty($propertyName);
        }

        return $this->reflection_cache[$class][$propertyName];
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    private function formatValue(mixed $value): mixed
    {
        return $value instanceof DateTimeInterface ? $value->format('Y-m-d H:i:s.v') : $value;
    }

    private function resolveTimestamp(object $object, AuditActionType $action): DateTimeInterface
    {
        $refClass = $this->getReflectionClass($object);

        $properties = $refClass->getProperties();

        foreach($properties as $property) {

            $attrs = $property->getAttributes(AuditTimestamp::class);

            if($attrs === []) {
                continue;
            }

            foreach($attrs as $attr) {

                /** @var AuditTimestamp $meta */
                $meta = $attr->newInstance();

                if($meta->action === $action) {
                    if (!$property->isPublic()) {
                        $property->setAccessible(true);
                    }
                    $value = $property->getValue($object);
                    if ($value instanceof DateTimeInterface) {
                        return $value;
                    }
                }
            }
        }

        return new DateTimeImmutable();
    }

    private function resolveActor(object $object, AuditActionType $action): string
    {
        $refClass = $this->getReflectionClass($object);

        $properties = $refClass->getProperties();

        foreach($properties as $property) {

            $attrs = $property->getAttributes(AuditActor::class);

            if($attrs === []) {
                continue;
            }

            foreach($attrs as $attr) {

                /** @var AuditActor $meta */
                $meta = $attr->newInstance();

                if($meta->action === $action) {
                    if (!$property->isPublic()) {
                        $property->setAccessible(true);
                    }
                    $value = $property->getValue($object);
                    if (is_string($value)) {
                        return $value;
                    }
                }
            }
        }
        return 'NOT RECORDED';
    }

    /**
     * @throws ReflectionException
     */
    private function shouldIncludeField(object $object, $meta, string $key): ?array
    {
        if (!$meta->hasField($key) && !$meta->hasAssociation($key)) {
            return null;
        }
        $property   = $this->getReflectionProperty($object, $key);
        $attributes = array_map(fn($a) => $a->getName(), $property->getAttributes());

        foreach ($property->getAttributes() as $attr) {
            if (is_a($attr->getName(), NotLogged::class, true)) {
                return null;
            }
        }

        $isJoin  = in_array(JoinColumn::class, $attributes, true);
        $mapKey  = $isJoin ? $key . '_id' : $key;
        return ['isJoin' => $isJoin, 'mapKey' => $mapKey];
    }

    private function setOriginalUpdated(
        array &$original,
        array &$updated,
        string $mapKey,
        bool $isJoin,
        mixed $old,
        mixed $new,
        AuditActionType $action
    ): void {
        // normalize values
        $oldVal = $isJoin ? (is_object($old) ? $old->id : null) : $this->formatValue($old);
        $newVal = $isJoin ? (is_object($new) ? $new->id : null) : $this->formatValue($new);

        $original[$mapKey] = $oldVal;
        $updated[$mapKey]  = ($action === AuditActionType::DELETE) ? null : $newVal;
    }

}
