<?php

declare(strict_types=1);

namespace Atlas\AuditBundle\Twig\Component;

use Atlas\AuditBundle\Repository\SystemRepository;
use Modules\Paginator\Service\Paginator;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent(name: 'audit_viewer', template: '@Audit/components/audit_viewer.html.twig')]
final class AuditViewerComponent
{
    use DefaultActionTrait;

    #[LiveProp]
    public string $table_name;

    #[LiveProp]
    public int $identifier;

    #[LiveProp]
    public int $per_page = 50;

    #[LiveProp(writable: true)]
    public int $page = 1;

    private ?array $page_cache = null;

    public function __construct(
        private readonly SystemRepository $audits,
        private readonly Paginator $paginator,
    ) {}

    public function getAuditsPage(): array
    {
        if ($this->page_cache !== null) {
            return $this->page_cache;
        }

        $query = $this->audits
            ->findForSubjectQueryBuilder($this->table_name, $this->identifier);

        return $this->page_cache = $this->paginator->paginate(
            $query,
            max(1, $this->page),
            max(1, $this->per_page),
        );
    }

    public function getRows(): array
    {
        $rows = [];

        foreach ($this->getAuditsPage()['items'] as $audit) {
            $rows[] = [
                'action'    => $audit->action,
                'user'      => $audit->user,
                'timestamp' => $audit->timestamp,
                'original'  => $audit->original_values_array,
                'changed'   => $audit->changed_values_array,
            ];
        }

        return $rows;
    }

    #[LiveAction]
    public function gotoPage(#[LiveArg] int $page): void
    {
        $this->page = max(1, $page);
        $this->page_cache = null;
    }

    #[LiveAction]
    public function nextPage(): void
    {
        $this->page++;
        $this->page_cache = null;
    }

    #[LiveAction]
    public function prevPage(): void
    {
        $this->page = max(1, $this->page - 1);
        $this->page_cache = null;
    }
}