<?php

declare(strict_types=1);

namespace Atlas\AuditBundle\Tests\Entity;

use Atlas\AuditBundle\Entity\System;
use PHPUnit\Framework\TestCase;

final class SystemTest extends TestCase
{
    public function testConstructStoresJsonAndTimestamp(): void
    {
        $s = new System(
            identifier: 123,
            tableName: 'users',
            action: System::INSERT,
            reason: 'some valid reason',
            actionBy: 'someone@example.com',
            originalValues: ['a' => 1],
            changedValues: ['a' => 2],
        );

        // Accessor should return label (Insert/Update/Delete)
        self::assertSame('Insert', $s->action);

        self::assertNotEmpty($s->original_values);
        self::assertNotEmpty($s->changed_values);
        self::assertNotEmpty($s->timestamp);
        self::assertSame('someone@example.com', $s->user);
        self::assertSame('users', $s->table_name);
        self::assertSame(123, $s->identifier);

        self::assertSame(['a' => 1], json_decode($s->original_values, true, flags: JSON_THROW_ON_ERROR));
        self::assertSame(['a' => 2], json_decode($s->changed_values, true, flags: JSON_THROW_ON_ERROR));
    }

    public function testConstructRejectsBadAction(): void
    {
        $this->expectException(\InvalidArgumentException::class);

        new System(
            identifier: 1,
            tableName: 't',
            action: 'X',
            reason: 'valid reason',
            actionBy: 'user',
            originalValues: ['x' => 1],
            changedValues: ['x' => 2],
        );
    }

    public function testConstructRejectsEmptyOriginalOrChangedValues(): void
    {
        $this->expectException(\InvalidArgumentException::class);

        new System(
            identifier: 1,
            tableName: 't',
            action: System::INSERT,
            reason: 'valid reason',
            actionBy: 'user',
            originalValues: [],
            changedValues: ['x' => 2],
        );
    }
}
