<?php

namespace Atlas\AuditBundle\Entity;

use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\AuditBundle\Repository\SystemRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use InvalidArgumentException;
use Symfony\Component\Validator\Constraints\Length;

#[NotLogged]
#[ORM\Entity(repositoryClass: SystemRepository::class)]
#[ORM\Table(name: 'audit_system')]
class System
{
    public const string INSERT = 'I';
    public const string UPDATE = 'U';
    public const string DELETE = 'D';

    public const array ACTIONS = [
        'I' => 'Insert',
        'U' => 'Update',
        'D' => 'Delete'
    ];

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column(nullable: true)]
    private(set) int $identifier;

    #[ORM\Column(length: 1)]
    private(set) string $action {
        get {
            return self::ACTIONS[$this->action];
        }
    }

    #[ORM\Column(length: 35)]
    #[Length(min: 2, max: 35)]
    private(set) string $table_name;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private(set) ?string $original_values;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private(set) ?string $changed_values;

    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    private(set) string $reason;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    private(set) DateTimeInterface $timestamp;

    #[ORM\Column(name: '`user`', length: 255)]
    private(set) string $user;

    public function __construct(
        int $identifier,
        string $tableName,
        string $action,
        string $reason,
        string $actionBy,
        ?array $originalValues = null,
        ?array $changedValues = null,
        ?DateTimeInterface $timestamp = null
    ) {
        //Automatically set the timestamp if none given
        $timestamp = $timestamp ?? new DateTimeImmutable();

        if (empty (mb_trim($action, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Action must be I = Insert, U = Update or D = Delete');
        }

        if ($action != self::INSERT && $action != self::DELETE && $action != self::UPDATE) {
            throw new InvalidArgumentException('Action must be I = Insert, U = Update or D = Delete');
        }

        if (empty (mb_trim($tableName, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Table name must have a value');
        }

        if (empty($identifier)) {
            throw new InvalidArgumentException('Identifier must be set');
        }

        if (empty (mb_trim($reason, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Reason must have a value');
        }

        if (empty (mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new InvalidArgumentException('Action by must have a value');
        }

        if (empty($originalValues)) {
            throw new InvalidArgumentException('Original values must be an array with at least 1 item');
        }

        if (empty($changedValues)) {
            throw new InvalidArgumentException('Changed values must be an array with at least 1 item');
        }

        $this->identifier = $identifier;
        $this->table_name = $tableName;
        $this->action = $action;
        $this->reason = $reason;
        $this->user = $actionBy;
        $this->original_values = json_encode($originalValues);
        $this->changed_values = json_encode($changedValues);
        $this->timestamp = $timestamp;
    }
}
