<?php

declare(strict_types=1);

namespace Atlas\AuditBundle\Tests\Listener;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\AuditBundle\Listener\LoggerListener;
use DateTimeImmutable;
use Doctrine\Persistence\ManagerRegistry;
use Doctrine\Persistence\ObjectManager;
use PHPUnit\Framework\Attributes\PreserveGlobalState;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use ReflectionException;
use ReflectionMethod;

#[PreserveGlobalState(false)]
final class LoggerListenerTest extends TestCase
{
    private function makeListener(string $connections = ''): LoggerListener
    {
        $logger = $this->createStub(LoggerInterface::class);
        $registry = $this->createStub(ManagerRegistry::class);

        return new LoggerListener($logger, $registry, $connections);
    }

    /**
     * @throws ReflectionException
     */
    private function invokePrivate(object $obj, string $method, array $args): mixed
    {
        $ref = new ReflectionMethod($obj, $method);
        return $ref->invokeArgs($obj, $args);
    }

    /**
     * @throws ReflectionException
     */
    public function testResolveActorReturnsStringForMatchingAction(): void
    {
        $entity = new class {
            #[AuditActor(action: AuditActionType::INSERT)]
            private string $actor = 'matt';
        };

        $listener = $this->makeListener();

        $actor = $this->invokePrivate($listener, 'resolveActor', [$entity, AuditActionType::INSERT]);
        self::assertSame('matt', $actor);

        $actor2 = $this->invokePrivate($listener, 'resolveActor', [$entity, AuditActionType::UPDATE]);
        self::assertSame('NOT RECORDED', $actor2);
    }

    /**
     * @throws ReflectionException
     */
    public function testResolveTimestampReturnsDateTimeForMatchingAction(): void
    {
        $dt = new DateTimeImmutable('2025-12-12 08:00:00');

        $entity = new class($dt) {
            public function __construct(
                #[AuditTimestamp(action: AuditActionType::UPDATE)]
                private DateTimeImmutable $ts
            ) {}
        };

        $listener = $this->makeListener();

        $ts = $this->invokePrivate($listener, 'resolveTimestamp', [$entity, AuditActionType::UPDATE]);
        self::assertInstanceOf(\DateTimeInterface::class, $ts);
        self::assertSame($dt->format('c'), $ts->format('c'));
    }

    /**
     * @throws ReflectionException
     */
    public function testHandleEventSkipsNotLoggedClass(): void
    {
        $entity = new #[NotLogged] class {};

        $listener = $this->makeListener();

        $em = $this->createMock(ObjectManager::class);
        // If NotLogged works, it must return before touching the EM.
        $em->expects(self::never())->method('getClassMetadata');

        $this->invokePrivate($listener, 'handleEvent', [$em, $entity, AuditActionType::INSERT]);

        self::assertTrue(true);
    }
}
