<?php
declare(strict_types=1);

namespace Atlas\ExportBundle\Tests\Service;

use Atlas\ExportBundle\Service\CsvBuilder;
use Atlas\ExportBundle\Service\DownloadResponseFactory;
use Atlas\ExportBundle\Service\Exporter;
use Atlas\ExportBundle\Service\SpreadsheetBuilder;
use Atlas\ExportBundle\Service\SpssBuilder;
use Atlas\ExportBundle\Service\SpssFormSpecificationCompiler;
use PHPUnit\Framework\TestCase;

final class ExporterTest extends TestCase
{
    private function exporter(): Exporter
    {
        return new Exporter(
            spreadsheetBuilder: new SpreadsheetBuilder(),
            csvBuilder: new CsvBuilder(),
            spssFormSpecificationCompiler: new SpssFormSpecificationCompiler(),
            spssBuilder: new SpssBuilder(),
            downloadResponseFactory: new DownloadResponseFactory(),
        );
    }

    public function testCsvResponse(): void
    {
        $resp = $this->exporter()->csv([['a' => 1]]);
        self::assertStringContainsString('text/csv', (string) $resp->headers->get('Content-Type'));
        self::assertStringContainsString('export.csv', (string) $resp->headers->get('Content-Disposition'));
    }

    public function testExcelResponse(): void
    {
        $resp = $this->exporter()->excel([['a' => 1]]);
        self::assertStringContainsString(
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            (string) $resp->headers->get('Content-Type')
        );
        self::assertStringContainsString('export.xlsx', (string) $resp->headers->get('Content-Disposition'));
        self::assertSame('PK', substr((string) $resp->getContent(), 0, 2));
    }

    public function testSpssResponseWithManualMetadata(): void
    {
        $resp = $this->exporter()->spss(
            dataRows: [['age' => 10]],
            formSpec: null,
            manualMetadata: [
                'variables' => [
                    ['name' => 'age', 'label' => 'Age', 'type' => 'INTEGER'],
                ],
                'categoricalOptions' => [],
            ]
        );

        self::assertSame('application/zip', $resp->headers->get('Content-Type'));
        self::assertStringContainsString('export_spss.zip', (string) $resp->headers->get('Content-Disposition'));
        self::assertSame('PK', substr((string) $resp->getContent(), 0, 2));
    }
}