<?php

declare(strict_types=1);

namespace Atlas\ExportBundle\Service;

use Symfony\Component\HttpFoundation\Response;

/**
 * Public facade: controllers call this.
 */
final readonly class Exporter
{
    public function __construct(
        private SpreadsheetBuilder $spreadsheetBuilder,
        private CsvBuilder $csvBuilder,
        private SpssFormSpecificationCompiler $spssFormSpecificationCompiler,
        private SpssBuilder $spssBuilder,
        private DownloadResponseFactory $downloadResponseFactory,
    ) {
    }

    /**
     * Create an Excel .xlsx download response.
     *
     * @param array<int, array<string, scalar|null>> $data
     */
    public function excel(array $data): Response
    {
        $binary = $this->spreadsheetBuilder->buildXlsx($data);

        return $this->downloadResponseFactory->make(
            binary: $binary,
            filename: 'export.xlsx',
            contentType: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        );
    }

    /**
     * Create a CSV download response.
     *
     * @param array<int, array<string, scalar|null>> $data
     */
    public function csv(array $data): Response
    {
        $csv = $this->csvBuilder->buildCsv($data);

        return $this->downloadResponseFactory->make(
            binary: $csv,
            filename: 'export.csv',
            contentType: 'text/csv; charset=UTF-8'
        );
    }

    /**
     * Create an SPSS import package (.zip with data.tsv + import.sps).
     *
     * You can drive this three different ways:
     *
     * 1. Spec-driven (Phoenix CRF forms):
     *    -> pass $formSpec (big array like you showed)
     *    -> $manualMetadata = null
     *
     * 2. Manual metadata override (e.g. randomisation):
     *    -> pass $manualMetadata with keys 'variables' and 'categoricalOptions'
     *    -> $formSpec = null
     *
     * 3. Fallback / generic:
     *    -> both null, we infer TEXT columns from $dataRows
     *
     * @param array<int, array<string, scalar|null>> $dataRows
     * @param array<string,mixed>|null               $formSpec
     * @param array{
     *   variables: array<int, array{
     *     name:string,
     *     label:string,
     *     type:string,
     *     width?:int,
     *     decimals?:int,
     *     missing_values?:array<int,string>
     *   }>,
     *   categoricalOptions: array<string, array<string,string>>
     * }|null $manualMetadata
     */
    public function spss(
        array $dataRows,
        ?array $formSpec = null,
        ?array $manualMetadata = null
    ): Response {
        if ($formSpec !== null) {
            // 1. Spec-driven (full CRF spec -> compile)
            $compiled = $this->spssFormSpecificationCompiler->compile($formSpec);
            $variables = $compiled['variables'];
            $categoricalOptions = $compiled['categoricalOptions'];

        } elseif ($manualMetadata !== null) {
            // 2. Manual metadata provided by caller
            $variables = $manualMetadata['variables'] ?? [];
            $categoricalOptions = $manualMetadata['categoricalOptions'] ?? [];

        } else {
            // 3. Generic fallback, infer from first row
            $firstRow = $dataRows[0] ?? [];
            $variables = [];
            foreach (array_keys($firstRow) as $colName) {
                $variables[] = [
                    'name' => $colName,
                    'label' => $colName,
                    'type' => 'TEXT',
                    'width' => 255,
                ];
            }
            $categoricalOptions = [];
        }

        $package = $this->spssBuilder->buildSpssPackage(
            dataRows: $dataRows,
            variables: $variables,
            categoricalOptions: $categoricalOptions,
        );

        return $this->downloadResponseFactory->make(
            binary: $package['binary'],
            filename: $package['filename'],
            contentType: $package['contentType']
        );
    }
}
