<?php

declare(strict_types=1);

namespace Atlas\ExportBundle\Service;

/**
 * Takes a Phoenix-style form spec array and compiles it into
 * structured SPSS metadata:
 *
 * - variables: list of columns with name, label, type, width, decimals,
 *   and any "missing"/"skip" codes.
 *
 * - categoricalOptions: map of varName => [code => meaning] for
 *   category / yesno style fields.
 */
final readonly class SpssFormSpecificationCompiler
{
    /**
     * @param array $formSpec Full form spec config array
     *
     * @return array{
     *   variables: array<int, array{
     *     name:string,
     *     label:string,
     *     type:string,
     *     width?:int,
     *     decimals?:int,
     *     missing_values?:array<int,string>
     *   }>,
     *   categoricalOptions: array<string, array<string,string>>
     * }
     */
    public function compile(array $formSpec): array
    {
        $variables = [];
        $categoricalOptions = [];

        if (!isset($formSpec['elements']) || !\is_array($formSpec['elements'])) {
            return [
                'variables' => $variables,
                'categoricalOptions' => $categoricalOptions,
            ];
        }

        foreach ($formSpec['elements'] as $element) {
            if (!isset($element['spec']) || !\is_array($element['spec'])) {
                continue;
            }

            $spec = $element['spec'];

            // name (machine name)
            $name = $spec['name'] ?? null;
            if (!$name || !\is_string($name)) {
                continue;
            }

            // human-readable label
            $label = $spec['options']['label'] ?? $name;
            if (!\is_string($label)) {
                $label = $name;
            }

            // raw control type in the form spec (integer, yesno, category, date, textarea, decimal, etc)
            $rawType = strtolower((string) ($spec['type'] ?? 'text'));

            // infer SPSS logical type + shape
            [$logicalType, $width, $decimals] = $this->mapType($rawType, $spec);

            // collect any special sentinel codes
            $specialMissing = [];
            if (isset($spec['attributes']) && \is_array($spec['attributes'])) {
                foreach (['missing-code', 'skip-code'] as $codeKey) {
                    if (array_key_exists($codeKey, $spec['attributes'])) {
                        $specialMissing[] = (string)$spec['attributes'][$codeKey];
                    }
                }
            }

            $varDef = array_filter([
                'name' => $name,
                'label' => $label,
                'type' => $logicalType, // INTEGER | DECIMAL | CATEGORY | DATE | TEXT
                'width' => $width,
                'decimals' => $decimals,
                'missing_values' => $specialMissing,
            ], static fn ($v) => $v !== null && $v !== []);

            $variables[] = $varDef;

            // map coded answers (value_options) into categorical labels
            if (isset($spec['options']['value_options']) && \is_array($spec['options']['value_options'])) {
                $categoricalOptions[$name] = [];
                foreach ($spec['options']['value_options'] as $code => $meaning) {
                    $categoricalOptions[$name][(string)$code] = (string)$meaning;
                }
            } elseif ($rawType === 'yesno') {
                // implicit yes/no even if no explicit value_options
                $categoricalOptions[$name] = [
                    '1' => 'Yes',
                    '0' => 'No',
                ];
            }
        }

        return [
            'variables' => $variables,
            'categoricalOptions' => $categoricalOptions,
        ];
    }

    /**
     * Infer SPSS-ish type info from the form control type.
     *
     * @return array{0:string,1:int|null,2:int|null}
     *         [logicalType,width,decimals]
     */
    private function mapType(string $rawType, array $spec): array
    {
        $guessedWidth = 255;
        $guessedDecimals = null;

        switch ($rawType) {
            case 'integer':
                // whole number
                return ['INTEGER', null, 0];

            case 'decimal':
                // numeric with fractional part
                if (isset($spec['attributes']['data-inputmask-digits'])) {
                    $digits = (int) $spec['attributes']['data-inputmask-digits'];
                    $guessedDecimals = $digits;
                } else {
                    $guessedDecimals = 2;
                }
                return ['DECIMAL', null, $guessedDecimals];

            case 'yesno':
            case 'category':
                // coded choice field
                return ['CATEGORY', null, 0];

            case 'date':
                // we'll treat dates as string A11 on load ("DD/MM/YYYY")
                return ['DATE', null, null];

            case 'textarea':
            case 'text':
            default:
                // textual free entry
                return ['TEXT', $guessedWidth, null];
        }
    }
}
