<?php

namespace Atlas\ImpBundle\Service\Allocation;

use Atlas\ImpBundle\Contract\AlgorithmInterface;
use Atlas\ImpBundle\Domain\Enum\AllocationMethod;
use Atlas\ImpBundle\Domain\Specification;
use Atlas\ImpBundle\Entity\Allocation\Allocation;
use Atlas\ImpBundle\Repository\Allocation\BlockRepository;
use Atlas\ImpBundle\Repository\Allocation\SequenceRepository;
use DateTimeInterface;
use Doctrine\ORM\EntityManagerInterface;
use Random\RandomException;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Uid\UuidV7;

final readonly class Allocator
{

    /**
     * @param EntityManagerInterface $entity_manager
     * @param BlockRepository $blocks
     * @param SequenceRepository $sequences
     * @param AllocationAuditor $logger
     */
    public function __construct(
        private EntityManagerInterface $entity_manager,
        private BlockRepository $blocks,
        private SequenceRepository $sequences,
        private AllocationAuditor $logger
    ) {}

    /**
     * @param string $studyCode
     * @param Specification $spec
     * @param string $participantIdentifier
     * @param string $locationCode
     * @param array $variables
     * @param string $actionBy
     * @param DateTimeInterface|null $simulate
     * @param Uuid|null $simulationId
     * @return Allocation|false
     * @throws RandomException
     */
    public function allocate(
        string $studyCode,
        Specification $spec,
        string $participantIdentifier,
        string $locationCode,
        array $variables,
        string $actionBy,
        ?DateTimeInterface $simulate = null,
        ?Uuid $simulationId = null
    ): Allocation|false
    {

        $uri = new UuidV7();

        $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Allocator starting for IMP %s', $spec->imp_code), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
        $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Input value location: %s ', $locationCode), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        $allocation = $variables[$spec->allocation_var] ?? null;

        if($allocation === null) {
            $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, 'Allocator failed no allocation found', $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
            return false;
        }

        $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Input value allocation: %s ', $allocation), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        $matchedVars = [];

        /*
            format is:
            [
                'name' => 'var_1',
                'variable' => 'name_from_variable_array',
                'required' => true
            ]
        */
        foreach($spec->variables as $var) {

            $v = $variables[$var->variable] ?? null;
            $required = $var->required ?? false;

            if($required && $v === null) {
                $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Required variable not found: %s - %s', $var->name, $var->variable), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

                return false;
            }
            if(! $required && $v === null) {
                $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('None required variable not found: %s - %s', $var->name, $var->variable), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

                continue;
            }

            $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Variable found: %s - %s - %s', $var->name, $var->variable, $v), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

            $matchedVars[$var->name] = $v;
        }

        $blocks = $this->blocks->findMatchingBlocks($studyCode, $spec->imp_code, $matchedVars);

        if($blocks === []) {
            $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, 'No blocks found', $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
            return false;
        }

        $ids = [];

        foreach($blocks as $block) {
            $ids[] = $block->id;
        }

        $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Blocks found: %s', implode(',', $ids)), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        if($spec->configuration->method === AllocationMethod::RANDOM) {
            $sequences = $this->sequences->findUnallocatedSequences($studyCode, $spec->imp_code, $blocks, $allocation, locationCode: $locationCode, location: $spec->configuration->location, status: $spec->configuration->status);
            if(empty($sequences)) {
                $sequence = null;
            }
            else {
                $idx = random_int(0, count($sequences) - 1);
                $sequence = $sequences[$idx];
            }
        }
        else { //top 1
            $sequence = $this->sequences->findFirstUnallocatedSequence($studyCode, $spec->imp_code, $blocks, $allocation, locationCode: $locationCode, location: $spec->configuration->location, status: $spec->configuration->status);
        }

        if($sequence === null) {
            $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, 'No sequence found from sequences', $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
            return false;
        }

        $this->logger->log($studyCode, $spec->imp_code, $participantIdentifier, sprintf('Pack found: %s', $sequence->pack->pack_identifier), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        $allocation = new Allocation($studyCode, $spec->imp_code, $participantIdentifier, $sequence->pack, $uri, $actionBy, simulation: $simulate, simulationId: $simulationId);
        $sequence->setAllocated($actionBy);
        $sequence->pack->setAllocate($actionBy);

        //save to database
        $this->entity_manager->persist($allocation);
        $this->entity_manager->flush();

        return $allocation;
    }

}
