<?php

namespace Atlas\ImpBundle\Repository\Imp;

use Atlas\ImpBundle\Entity\Imp\Enum\PackLocation;
use Atlas\ImpBundle\Entity\Imp\Enum\PackStatus;
use Atlas\ImpBundle\Entity\Imp\Pack;
use Atlas\ImpBundle\Exception\PackNotFoundException;
use Atlas\ImpBundle\Service\Specification\SpecificationLoader;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Pack>
 */
class PackRepository extends ServiceEntityRepository
{

    public function __construct(
        ManagerRegistry $registry,
        private SpecificationLoader $specification_loader
    )
    {
        parent::__construct($registry, Pack::class);
    }

    /**
     * @param string $packIdentifier
     * @return Pack
     */
    public function findByPackIdentifierOrThrow(string $packIdentifier): Pack
    {
        $pack = $this->findOneBy(
            [
                'pack_identifier' => $packIdentifier,
            ]
        );

        if ($pack === null) {
            throw PackNotFoundException::fromPackIdentifier($packIdentifier);
        }

        return $pack;
    }

    /**
     * @param int $id
     * @return Pack
     * @throws PackNotFoundException
     */
    public function findOrThrow(int $id): Pack
    {
        $pack = $this->find($id);
        if ($pack === null) {
            throw PackNotFoundException::fromId($id);
        }
        return $pack;
    }

    /**
     * Get all packs, or (if $locationCode is provided) only packs
     * with that location_code and no assigned PackLocation.
     *
     * @param string $studyCode
     * @param string $impCode
     * @param string $locationCode
     * @return Pack[]
     */
    public function findAllWithLocation(
        string $studyCode,
        string $impCode,
        string $locationCode
    ): array {
        $qb = $this->createQueryBuilder('p')
            ->andWhere('p.study_code = :studyCode')
            ->andWhere('p.imp_code = :impCode')
            ->andWhere('p.location_code = :locationCode')
            ->setParameter('studyCode', $studyCode)
            ->setParameter('impCode', $impCode)
            ->setParameter('locationCode', $locationCode)
            ->orderBy('p.id', 'ASC');

        return $qb->getQuery()->getResult();
    }

    /**
     * Generate counts for a given location.
     *
     * @param string $studyCode
     * @param string $impCode
     * @param string $locationCode
     *
     * @return array{
     *     location: string,
     *     count: int,
     *     available: int,
     *     unavailable: int,
     *     allocated: int,
     *     destroyed: int
     * }
     */
    public function generateLocationCounts(
        string $studyCode,
        string $impCode,
        string $locationCode
    ): array {
        $qb = $this->createQueryBuilder('p');

        $qb
            ->select(
                'COUNT(p.id) AS total',
                'SUM(CASE WHEN p.status = :statusIn THEN 1 ELSE 0 END) AS available',
                'SUM(CASE WHEN p.status = :statusOut THEN 1 ELSE 0 END) AS unavailable',
                'SUM(CASE WHEN p.status = :statusAllocated THEN 1 ELSE 0 END) AS allocated',
                'SUM(CASE WHEN p.status = :statusDestroy THEN 1 ELSE 0 END) AS destroyed'
            )
            ->where('p.study_code = :studyCode')
            ->andWhere('p.imp_code = :impCode')
            ->andWhere('p.location_code = :locationCode')
            ->setParameter('studyCode', $studyCode)
            ->setParameter('impCode', $impCode)
            ->setParameter('locationCode', $locationCode)
            ->setParameter('statusIn', PackStatus::IN)
            ->setParameter('statusOut', PackStatus::OUT)
            ->setParameter('statusAllocated', PackStatus::ALLOCATE)
            ->setParameter('statusDestroy', PackStatus::DESTROY);

        $result = $qb->getQuery()->getSingleResult();

        return [
            'location' => $locationCode,
            'count' => (int) $result['total'],
            'available' => (int) $result['available'],
            'unavailable' => (int) $result['unavailable'],
            'allocated' => (int) $result['allocated'],
            'destroyed' => (int) $result['destroyed'],
        ];
    }

    public function generateLabelCountsForLocation(
        string $studyCode,
        string $impCode,
        string $location_code,
        ?int $version = null,
        PackLocation $location = PackLocation::LOCATION,
        PackStatus $status = PackStatus::IN,
        bool $includeGlobal = true,
        bool $excludeExpired = true,
        ?DateTimeInterface $expiry = null,
    ): array
    {
        $expiry = $expiry ?? new DateTimeImmutable();

        $specification = $this->specification_loader->load(
            $studyCode,
            $impCode,
            version: $version
        );

        $labels = $specification->labels;

        $countsQb = $this->createQueryBuilder('p')
            ->select('p.label as label, COUNT(p.id) AS label_count')
            ->where('p.study_code = :studyCode')
            ->andWhere('p.imp_code = :impCode')
            ->andWhere('p.location = :location')
            ->andWhere('p.status = :status')
            ->andWhere('p.label IN (:labels)')
            ->groupBy('p.label')
            ->setParameter('studyCode', $studyCode)
            ->setParameter('impCode', $impCode)
            ->setParameter('labels', $labels)
            ->setParameter('location', $location)
            ->setParameter('status', $status);

        if($includeGlobal) {
            $countsQb->andWhere('(p.location_code = :location_code OR p.location_code IS NULL)')
                ->setParameter('location_code', $location_code);
        }
        else {
            $countsQb->andWhere('(p.location_code = :location_code)')
                ->setParameter('location_code', $location_code);
        }

        if($excludeExpired) {
            $countsQb->andWhere('(p.pack_expiry IS NULL OR p.pack_expiry >= :now)')
                ->setParameter('now', $expiry);
        }

        /** @var array<int, array{label: string, label_count: string|int}> $countRows */
        $countRows = $countsQb->getQuery()->getArrayResult();

        $result = array_fill_keys($labels, 0);

        foreach ($countRows as $row) {
            $result[$row['label']] = (int)$row['label_count'];
        }

        return $result;
    }
}
