<?php

namespace Atlas\ImpBundle\Service\Specification;

use Atlas\ImpBundle\Domain\Specification;
use Atlas\ImpBundle\Exception\SpecificationException;
use Psr\Cache\CacheItemPoolInterface;
use Psr\Cache\InvalidArgumentException;
use Throwable;

readonly class SpecificationCache
{
    public function __construct(
        private CacheItemPoolInterface $cache
    )
    {
    }

    /**
     * @param string $studyCode
     * @param string $impName
     * @param int $version
     * @param string $path
     * @return Specification
     * @throws InvalidArgumentException
     * @throws SpecificationException
     */
    public function load(string $studyCode, string $impName, int $version, string $path): Specification
    {
        $hash = @hash_file('sha256', $path);

        if (!$hash) {
            throw new SpecificationException(sprintf(
                'Specification for %s %s v%d could not be hashed.',
                $studyCode,
                $impName,
                $version
            ));
        }

        $key = sprintf('specification.%s.%s.%d.%s', $studyCode, $impName, $version, $hash);

        $key = preg_replace('/[^A-Za-z0-9_.:]/u', '_', $key) ?? $key;

        $cached = $this->cache->getItem($key);
        if ($cached->isHit()) {
            return $cached->get();
        }

        try {

            $specificationArray = (static function (string $__file): array {
                return include $__file;
            })($path);

        } catch (Throwable $e) {
            throw new SpecificationException('Incorrect specification file format.', code: 0, previous: $e);
        }

        $parsed = new Specification($specificationArray['imp_code'], $specificationArray['allocation_var'], $specificationArray['allocations'], $specificationArray['variables'], $specificationArray['labels']);

        $cached->set($parsed);

        try {
            $saved = $this->cache->save($cached);

            if (!$saved) {
                throw new SpecificationException('Could not save specification cache.');
            }
        } catch (InvalidArgumentException $e) {
            // rethrow with context (or log then rethrow)
            throw new SpecificationException(sprintf('Cache save failed for specification %s: %s', $key, $e->getMessage()), 0, $e);
        }

        return $parsed;
    }
}
