<?php

namespace Atlas\ImpBundle\Service\Specification;

use Atlas\ImpBundle\Domain\Specification;
use Atlas\ImpBundle\Exception\SpecificationException;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

readonly class SpecificationLoader
{
    public function __construct(
        #[Autowire(param: 'shared.imp.spec_root')]
        private string $base_dir,
        private SpecificationCache $cache
    ) {}

    /**
     * @param string $studyCode
     * @param string $impName
     * @param ?int $version
     * @return Specification
     * @throws InvalidArgumentException
     */
    public function load(string $studyCode, string $impName, ?int $version = null): Specification
    {
        if($version === null) {
            [ $version, $path ] = $this->getVersion($studyCode, $impName);
        }
        else {
            $path = $this->getPath($studyCode, $impName, $version);
        }

        return $this->cache->load($studyCode, $impName, $version, $path);
    }

    /**
     * @param string $studyCode
     * @param string $impName
     * @param int $version
     * @return string
     */
    private function getPath(string $studyCode, string $impName, int $version): string
    {
        $dir = rtrim($this->base_dir, '/');

        $candidates = [
            sprintf('%s/%s/v%d/%s.php', $dir, strtoupper($studyCode), $version, strtoupper($impName)),
            sprintf('%s/%s/v%d/%s.php', $dir, strtolower($studyCode), $version, strtolower($impName)),
        ];

        $errors = [];

        foreach ($candidates as $path) {
            if (!is_file($path)) {
                $errors[] = sprintf('Not found: %s', $path);
                continue;
            }
            if (!is_readable($path)) {
                $errors[] = sprintf('Not readable: %s', $path);
                continue;
            }
            return $path;
        }

        throw new SpecificationException(sprintf(
            'Specification for %s %s v%d could not be located: %s',
            $studyCode,
            $impName,
            $version,
            implode('; ', $errors)
        ));
    }

    /**
     * @param string $studyCode
     * @param string $impName
     * @return array
     */
    private function getVersion(string $studyCode, string $impName): array
    {
        $dir = rtrim($this->base_dir, '/');

        $candidates = [
            sprintf('%s/%s', $dir, strtoupper($studyCode)),
            sprintf('%s/%s', $dir, strtolower($studyCode)),
        ];

        $versions = [];
        foreach ($candidates as $candidate) {
            if (!is_dir($candidate)) {
                continue;
            }

            $found = glob($candidate . '/v[0-9]*', GLOB_ONLYDIR);
            if ($found === false) {
                continue; // glob error
            }

            foreach ($found as $directory) {
                $version = ltrim(basename($directory), 'v');
                // Ensure it's only digits (e.g., not "v1-beta")
                if (ctype_digit($version)) {
                    $versions[(int) $version] = true; // Use keys for automatic uniqueness
                }
            }
        }

        $versions = array_keys($versions);

        if (empty($versions)) {
            throw new SpecificationException(sprintf(
                'No version directories (e.g., "v1", "v2") found for study %s',
                $studyCode
            ));
        }

        rsort($versions, flags: SORT_NUMERIC);

        foreach ($versions as $version) {
            try {
                return [ $version, $this->getPath($studyCode, $impName, $version) ];
            }
            catch(SpecificationException) {
                //do nothing
            }
        }

        throw new SpecificationException(sprintf(
            'No version directories (e.g., "v1", "v2") found for study %s and randomisation %s',
            $studyCode,
            $impName
        ));
    }
}