<?php

namespace Atlas\ImpBundle\Domain;

use Atlas\ImpBundle\Exception\SpecificationException;

final readonly class Specification
{
    private(set) string $imp_code;
    private(set) string $allocation_var;
    private(set) AllocationConfiguration $configuration;

    /** @var AlgorithmVariable[] */
    private(set) array $variables;

    /** @var string[] */
    private(set) array $labels;

    /** @var array<string, string> */
    private(set) array $label_map;

    public function __construct(
        string $impCode,
        string $allocationVar,
        array $allocations,
        array $variables,
        array $labels
    )
    {

        $impCode = mb_trim($impCode, encoding: 'UTF-8');
        if ($impCode === '') {
            throw new SpecificationException('imp_code must not be empty');
        }

        $this->imp_code = $impCode;

        $allocationVar = mb_trim($allocationVar, encoding: 'UTF-8');
        if ($allocationVar === '') {
            throw new SpecificationException('allocation_var must not be empty');
        }

        $this->allocation_var = $allocationVar;

        if($allocations === []){
            throw new SpecificationException('allocations[] configuration must not be empty');
        }

        if(! isset($allocations['method'])){
            throw new SpecificationException('allocations.method must not be empty');
        }

        if(! isset($allocations['status'])){
            throw new SpecificationException('allocations.status must not be empty');
        }

        if(! isset($allocations['location'])){
            throw new SpecificationException('allocations.location must not be empty');
        }

        $this->configuration = new AllocationConfiguration($allocations['method'], $allocations['status'], $allocations['location']);

        $vars = [];

        foreach($variables as $key => $variable){
            if($variable === []){
                throw new SpecificationException(sprintf('variables[%s] configuration must not be empty', $key));
            }

            if(! isset($variable['name'])){
                throw new SpecificationException(sprintf('variables[%s].name must not be empty', $key));
            }

            if(! isset($variable['variable'])){
                throw new SpecificationException(sprintf('variables[%s].variable must not be empty', $key));
            }

            $vars[] = new AlgorithmVariable($variable['name'], $variable['variable'], $variable['required'] ?? false, $key);
        }

        $this->variables = $vars;

        if(empty($labels)){
            throw new SpecificationException('labels must not be empty and list each IMP label');
        }

        $label_codes = [];
        $label_map = [];

        foreach ($labels as $key => $value) {
            // Format A: ['INT', 'PLC'] => $key is int, $value is code
            // Format B: ['INT' => 'Intervention'] => $key is code, $value is name

            if (is_int($key)) {
                $code = mb_trim((string) $value, encoding: 'UTF-8');
                if ($code === '') {
                    throw new SpecificationException(sprintf('labels[%s] label must not be empty', $key));
                }

                if(! in_array($code, $label_codes, true)) {
                    $label_codes[] = mb_strtoupper($code, encoding: 'UTF-8');
                    $label_map[$code] = $code; // fallback display name = code
                }

                continue;
            }

            $code = mb_trim((string) $key, encoding: 'UTF-8');
            if ($code === '') {
                throw new SpecificationException(sprintf('labels[%s] label code must not be empty', $key));
            }

            $name = mb_trim((string) $value, encoding: 'UTF-8');
            if ($name === '') {
                throw new SpecificationException(sprintf('labels[%s] label name must not be empty', $code));
            }

            if(! in_array($code, $label_codes, true)) {
                $label_codes[] = mb_strtoupper($code, encoding: 'UTF-8');
                $label_map[$code] = $name;
            }
        }

        $this->labels = $label_codes;
        $this->label_map = $label_map;
    }
}