<?php

namespace Atlas\ImpBundle\Service\Allocation;

use Atlas\ImpBundle\Domain\Enum\AllocationMethod;
use Atlas\ImpBundle\Domain\Specification;
use Atlas\ImpBundle\Entity\Imp\Allocation;
use Atlas\ImpBundle\Repository\Allocator\BlockRepository;
use Atlas\ImpBundle\Repository\Allocator\SequenceRepository;
use Atlas\ImpBundle\Service\Specification\SpecificationLoader;
use DateTimeInterface;
use Doctrine\ORM\EntityManagerInterface;
use Random\RandomException;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Uid\UuidV7;

final readonly class Allocator
{

    /**
     * @param EntityManagerInterface $entity_manager
     * @param BlockRepository $blocks
     * @param SequenceRepository $sequences
     * @param SpecificationLoader $loader
     * @param AllocationAuditor $logger
     */
    public function __construct(
        private EntityManagerInterface $entity_manager,
        private BlockRepository $blocks,
        private SequenceRepository $sequences,
        private SpecificationLoader $loader,
        private AllocationAuditor $logger
    ) {}

    /**
     * @param string $studyCode
     * @param string $impName
     * @param string $location
     * @param int $version
     * @param string $participantIdentifier
     * @param string $actionBy
     * @param array $variables
     * @param DateTimeInterface|null $simulate
     * @param Uuid|null $simulationId
     * @param Specification|null $specification
     * @return Allocation|false
     * @throws RandomException
     */
    public function allocate(
        string $studyCode,
        string $impName,
        string $location,
        int $version,
        string $participantIdentifier,
        string $actionBy,
        array $variables = [],
        ?DateTimeInterface $simulate = null,
        ?Uuid $simulationId = null,
        ?Specification $specification = null
    ): Allocation|false
    {

        $studyCode = mb_strtoupper($studyCode, encoding: 'utf-8');

        $specification = $specification ?? $this->loader->load($studyCode, $impName, $version);

        $uri = new UuidV7();

        $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Allocator starting for IMP %s', $specification->imp_code), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
        $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Input value location: %s ', $location), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        $allocation = $variables[$specification->allocation_var] ?? null;

        if($allocation === null) {
            $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, 'Allocator failed no allocation found', $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
            return false;
        }

        $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Input value allocation: %s ', $allocation), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        $matchedVars = [];

        /*
            format is:
            [
                'name' => 'var_1',
                'variable' => 'name_from_variable_array',
                'required' => true
            ]
        */
        foreach($specification->variables as $var) {

            $v = $variables[$var->variable] ?? null;
            $required = $var->required ?? false;

            if($required && $v === null) {
                $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Required variable not found: %s - %s', $var->name, $var->variable), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

                return false;
            }
            if(! $required && $v === null) {
                $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('None required variable not found: %s - %s', $var->name, $var->variable), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

                continue;
            }

            $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Variable found: %s - %s - %s', $var->name, $var->variable, $v), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

            $matchedVars[$var->name] = $v;
        }

        $blocks = $this->blocks->findMatchingBlocks($studyCode, $specification->imp_code, $matchedVars);

        if($blocks === []) {
            $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, 'No blocks found', $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
            return false;
        }

        $ids = [];

        foreach($blocks as $block) {
            $ids[] = $block->id;
        }

        $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Blocks found: %s', implode(',', $ids)), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        if($specification->configuration->method === AllocationMethod::RANDOM) {
            $sequences = $this->sequences->findUnallocatedSequences($studyCode, $specification->imp_code, $blocks, $allocation, locationCode: $location, location: $specification->configuration->location, status: $specification->configuration->status);
            if(empty($sequences)) {
                $sequence = null;
            }
            else {
                $idx = random_int(0, count($sequences) - 1);
                $sequence = $sequences[$idx];
            }
        }
        else { //top 1
            $sequence = $this->sequences->findFirstUnallocatedSequence($studyCode, $specification->imp_code, $blocks, $allocation, locationCode: $location, location: $specification->configuration->location, status: $specification->configuration->status);
        }

        if($sequence === null) {
            $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, 'No sequence found from sequences', $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);
            return false;
        }

        $this->logger->log($studyCode, $specification->imp_code, $participantIdentifier, sprintf('Pack found: %s', $sequence->pack->pack_identifier), $actionBy, $uri, simulate: $simulate, simulateId: $simulationId);

        $allocation = new Allocation($studyCode, $specification->imp_code, $participantIdentifier, $sequence->pack, $version, $uri, $actionBy, simulation: $simulate, simulationId: $simulationId);
        $sequence->setAllocated($actionBy);
        $sequence->pack->setAllocate($actionBy);

        //save to database
        $this->entity_manager->persist($allocation);
        $this->entity_manager->flush();

        return $allocation;
    }

}
