<?php

declare(strict_types=1);

namespace Atlas\ImpBundle\Entity\Study;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\ImpBundle\Exception\ActivateException;
use Atlas\ImpBundle\Exception\NotBlankException;
use Atlas\ImpBundle\Exception\NoUpdateRequiredException;
use Atlas\ImpBundle\Repository\Study\ImpRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;

#[ORM\Entity(repositoryClass: ImpRepository::class)]
#[ORM\Table(name: 'sys_imp')]
#[UniqueEntity(fields: ['code'])]
class Imp
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[Length(min: 2, max: 255)]
    private(set) string $name;

    #[ORM\Column(length: 8, unique: true)]
    #[NotBlank]
    #[Length(min: 2, max: 8)]
    private(set) string $code;

    #[ORM\Column]
    private(set) bool $active = false;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $modified_by;

    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    #[NotBlank]
    private(set) string $reason;

    public function __construct(string $name, string $code, string $actionBy) {

        if (empty($name = mb_trim($name, encoding: 'UTF-8'))) {
            throw new NotBlankException('Name should not be empty');
        }

        if (empty($code = mb_trim($code, encoding: 'UTF-8'))) {
            throw new NotBlankException('Code should not be empty');
        }

        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        $this->name = $name;
        $this->code = strtoupper($code);
        $this->active = false;
        $this->created = $this->modified = new DateTimeImmutable();
        $this->created_by = $this->modified_by = $actionBy;
        $this->reason = 'Initial creation of IMP';
    }

    /**
     * @param string $actionBy
     * @param string $reason
     * @param string|null $name
     * @throws NotBlankException
     * @throws NoUpdateRequiredException
     */
    public function update(
        string $actionBy,
        string $reason,
        ?string $name = null,
    ): void {
        $timestamp = new DateTimeImmutable();

        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }

        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        $changed = false;

        if (!empty($name) && $this->name !== $name = mb_trim($name, encoding: 'UTF-8')) {
            $this->name = $name;
            $changed = true;
        }

        if ($changed) {
            $this->modified = $timestamp;
            $this->modified_by = $actionBy;
            $this->reason = $reason;
        }

        if(! $changed) throw NoUpdateRequiredException::forId('User', $this->id);
    }

    /**
     * @param string $actionBy
     * @param string $reason
     * @param bool $activate
     * @throws NotBlankException
     * @throws ActivateException
     */
    public function activate(string $actionBy, string $reason, bool $activate = true): void
    {
        if (empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw new NotBlankException('Reason should not be empty');
        }
        if (empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw new NotBlankException('Action by should not be empty');
        }

        $timestamp = new DateTimeImmutable();

        if ($this->is_active !== $activate) {
            $this->active = !$activate;
            $this->modified = $timestamp;
            $this->modified_by = $actionBy;
            $this->reason = $reason;
        }
        else {
            throw ActivateException::forResource('IMP', $activate);
        }
    }


    //virtual variables
    public bool $is_active {
        get => $this->active;
    }
}
