<?php

namespace Atlas\ImpBundle\Service\Imp;

use Atlas\ImpBundle\Dto\PackCountDto;
use Atlas\ImpBundle\Entity\Imp\Enum\PackStatus;
use Atlas\ImpBundle\Entity\Imp\Pack;
use Atlas\ImpBundle\Repository\Imp\PackRepository;
use Doctrine\ORM\EntityManagerInterface;

readonly class PackManager
{
    /**
     * @param EntityManagerInterface $entity_manager
     * @param PackRepository $packs
     */
    public function __construct(
        private EntityManagerInterface $entity_manager,
        private PackRepository $packs,
    ) {}

    public function getPackFromPackIdentifier(string $packIdentifier): Pack
    {
        return $this->packs->findByPackIdentifierOrThrow($packIdentifier);
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @param bool $in
     * @return void
     */
    public function receipt(int|Pack $pack, string $actionBy, bool $in = true): void
    {
        if(is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setReceipt($in ? PackStatus::IN : PackStatus::OUT, $actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function in(int|Pack $pack, string $actionBy): void
    {
        if(is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setIn($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function out(int|Pack $pack, string $actionBy): void
    {
        if(is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setout($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function dispense(int|Pack $pack, string $actionBy): void
    {
        if(is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setDispense($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function return(int|Pack $pack, string $actionBy): void
    {
        if(is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setReturn($actionBy);
        $this->entity_manager->flush();
    }
    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function destroy(int|Pack $pack, string $actionBy): void
    {
        if(is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setDestroy($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param $locationCode
     * @return PackCountDto
     */
    public function getLocationCounts($locationCode): PackCountDto
    {
        $result = $this->packs->generateLocationCounts($locationCode);

        return new PackCountDto(
            $result['location'],
            $result['count'],
            $result['available'],
            $result['deactivated'],
        );
    }
}
