<?php

namespace Atlas\ImpBundle\Repository\Imp;

use App\Entity\Form\Form;
use App\Exception\Form\FormNotFoundException;
use Atlas\ImpBundle\Entity\Imp\Enum\PackLocation;
use Atlas\ImpBundle\Entity\Imp\Enum\PackStatus;
use Atlas\ImpBundle\Entity\Imp\Pack;
use Atlas\ImpBundle\Exception\PackNotFoundException;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Pack>
 */
class PackRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Pack::class);
    }

    /**
     * @param string $packIdentifier
     * @return Pack
     */
    public function findByPackIdentifierOrThrow(string $packIdentifier): Pack
    {
        $pack = $this->findOneBy(
            [
                'pack_identifier' => $packIdentifier,
            ]
        );

        if ($pack === null) {
            throw PackNotFoundException::fromPackIdentifier($packIdentifier);
        }

        return $pack;
    }

    /**
     * @param int $id
     * @return Pack
     * @throws PackNotFoundException
     */
    public function findOrThrow(int $id): Pack
    {
        $pack = $this->find($id);
        if ($pack === null) {
            throw PackNotFoundException::fromId($id);
        }
        return $pack;
    }

    /**
     * Get all packs, or (if $locationCode is provided) only packs
     * with that location_code and no assigned PackLocation.
     *
     * @param string $studyCode
     * @param string $impCode
     * @param string $locationCode
     * @return Pack[]
     */
    public function findAllWithLocation(
        string $studyCode,
        string $impCode,
        string $locationCode
    ): array {
        $qb = $this->createQueryBuilder('p')
            ->andWhere('p.study_code = :studyCode')
            ->andWhere('p.imp_code = :impCode')
            ->andWhere('p.location_code = :locationCode')
            ->setParameter('studyCode', $studyCode)
            ->setParameter('impCode', $impCode)
            ->setParameter('locationCode', $locationCode)
            ->orderBy('p.id', 'ASC');

        return $qb->getQuery()->getResult();
    }

    /**
     * Generate counts for a given location.
     *
     * @param string $studyCode
     * @param string $impCode
     * @param string $locationCode
     *
     * @return array{
     *     location: string,
     *     count: int,
     *     available: int,
     *     unavailable: int,
     *     allocated: int,
     *     destroyed: int
     * }
     */
    public function generateLocationCounts(
        string $studyCode,
        string $impCode,
        string $locationCode
    ): array {
        $qb = $this->createQueryBuilder('p');

        $qb
            ->select(
                'COUNT(p.id) AS total',
                'SUM(CASE WHEN p.status = :statusIn THEN 1 ELSE 0 END) AS available',
                'SUM(CASE WHEN p.status = :statusOut THEN 1 ELSE 0 END) AS unavailable',
                'SUM(CASE WHEN p.status = :statusAllocated THEN 1 ELSE 0 END) AS allocated',
                'SUM(CASE WHEN p.status = :statusDestroy THEN 1 ELSE 0 END) AS destroyed'
            )
            ->where('p.study_code = :studyCode')
            ->andWhere('p.imp_code = :impCode')
            ->andWhere('p.location_code = :locationCode')
            ->setParameter('studyCode', $studyCode)
            ->setParameter('impCode', $impCode)
            ->setParameter('locationCode', $locationCode)
            ->setParameter('statusIn', PackStatus::IN)
            ->setParameter('statusOut', PackStatus::OUT)
            ->setParameter('statusAllocated', PackStatus::ALLOCATE)
            ->setParameter('statusDestroy', PackStatus::DESTROY);

        $result = $qb->getQuery()->getSingleResult();

        return [
            'location'     => $locationCode,
            'count'        => (int) $result['total'],
            'available'    => (int) $result['available'],
            'unavailable'  => (int) $result['unavailable'],
            'allocated'    => (int) $result['allocated'],
            'destroyed'    => (int) $result['destroyed'],
        ];
    }
}
