<?php

namespace Atlas\ImpBundle\Service\Allocation;

use Atlas\ImpBundle\Contract\AlgorithmInterface;
use Atlas\ImpBundle\Domain\Specification;
use Atlas\ImpBundle\Entity\Allocation\Allocation;
use Atlas\ImpBundle\Exception\AllocationException;
use DateTimeInterface;
use Symfony\Component\DependencyInjection\Attribute\AutowireIterator;
use Symfony\Component\Uid\Uuid;

class Allocator
{

    /**
     * @var AlgorithmInterface[]
     */
    private array $algorithms;

    /**
     * @param iterable $algorithms
     */
    public function __construct(
        #[AutowireIterator('app.allocation.algorithm')]
        iterable $algorithms
    ) {
        // Build a lookup map for fast access
        foreach ($algorithms as $algorithm) {
            $this->algorithms[$algorithm->getType()] = $algorithm;
        }
    }

    /**
     * @param string $studyCode
     * @param Specification $spec
     * @param string $participantIdentifier
     * @param string $locationCode
     * @param array $variables
     * @param string $actionBy
     * @param DateTimeInterface|null $simulate
     * @param Uuid|null $simulationId
     * @return Allocation|false
     */
    public function allocate(string $studyCode, Specification $spec, string $participantIdentifier, string $locationCode, array $variables, string $actionBy, ?DateTimeInterface $simulate = null, ?Uuid $simulationId = null): Allocation|false
    {
        $type = $spec['allocation']['type'] ?? null;

        if (!$type || !isset($this->algorithms[$type])) {
            throw new AllocationException("No algorithm available for type: $type");
        }

        return $this->algorithms[$type]->allocate($studyCode, $spec, $participantIdentifier, $locationCode, $variables, $actionBy, simulate: $simulate, simulationId: $simulationId);
    }

}
