<?php

declare(strict_types=1);

namespace Atlas\ImpBundle\Entity\Study;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\ImpBundle\Exception\ActivateException;
use Atlas\ImpBundle\Repository\Study\ImpRepository;
use Atlas\ImpBundle\Exception\NotBlankException;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;

#[ORM\Entity(repositoryClass: ImpRepository::class)]
#[ORM\Table(name: 'sys_imp')]
#[ORM\UniqueConstraint(columns: ['study_code', 'code'])]
#[UniqueEntity(fields: ['study_code', 'code'])]
class Imp
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[Length(min: 2, max: 255)]
    private(set) string $name;

    #[ORM\Column(length: 8)]
    #[NotBlank]
    #[Length(min: 2, max: 8)]
    private(set) string $code;

    #[ORM\Column(length: 10)]
    private(set) string $study_code;

    #[ORM\Column]
    private(set) bool $active = false;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $modified_by;

    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    #[NotBlank]
    private(set) string $reason;

    public function __construct(
        string $code,
        string $name,
        string $studyCode,
        string $actionBy,
        ?DateTimeInterface $timestamp = null
    ) {
        $code = mb_trim($code, encoding: 'UTF-8');
        $name = mb_trim($name, encoding: 'UTF-8');
        $studyCode = mb_trim($studyCode, encoding: 'UTF-8');
        $actionBy = mb_trim($actionBy, encoding: 'UTF-8');

        if ($code === '' || $name === '' || $studyCode === '' || $actionBy === '') {
            throw new NotBlankException('IMP fields must not be empty');
        }

        $this->code = mb_strtoupper($code, 'UTF-8');
        $this->name = $name;
        $this->study_code = $studyCode;
        $this->active = false;

        $this->created = $this->modified = $timestamp ?? new DateTimeImmutable();
        $this->created_by = $this->modified_by = $actionBy;
        $this->reason = 'Initial creation of IMP';
    }

    public function activate(string $actionBy, string $reason, bool $activate = true): void
    {
        $reason = mb_trim($reason, 'UTF-8');
        $actionBy = mb_trim($actionBy, 'UTF-8');

        if ($reason === '' || $actionBy === '') {
            throw new NotBlankException('Reason and actionBy must not be empty');
        }

        if ($this->active === $activate) {
            throw ActivateException::forResource('IMP', $activate);
        }

        $this->active = $activate;
        $this->modified = new DateTimeImmutable();
        $this->modified_by = $actionBy;
        $this->reason = $reason;
    }

    public bool $is_active {
        get => $this->active;
    }
}