<?php

declare(strict_types=1);

namespace Atlas\ImpBundle\Entity\Allocation;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\ImpBundle\Entity\Imp\Pack;
use Atlas\ImpBundle\Repository\Allocation\AllocationRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;

#[NotLogged]
#[ORM\Entity(repositoryClass: AllocationRepository::class)]
#[ORM\UniqueConstraint(columns: ['study_code', 'imp_code','participant_identifier'])]
#[ORM\Table(name: 'imp_allocation')]
class Allocation
{

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    private(set) string $study_code;

    #[ORM\Column]
    private(set) string $imp_code;

    #[ORM\Column]
    private(set) string $participant_identifier;

    #[ORM\OneToOne(targetEntity: Pack::class)]
    #[ORM\JoinColumn(name: 'pack_id', referencedColumnName: 'id', unique: true, nullable: false)]
    private(set) Pack $pack;

    #[ORM\Column(type: UuidType::NAME, nullable: true, options: ['default' => null])]
    private(set) ?Uuid $run_id = null;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $allocated;

    #[ORM\Column(length: 255)]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $allocated_by;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true, options: ['default' => null])]
    private(set) ?DateTimeInterface $simulation = null;

    #[ORM\Column(type: UuidType::NAME, nullable: true, options: ['default' => null])]
    private(set) ?Uuid $simulation_id = null;

    /**
     * @param string $studyCode
     * @param string $impCode
     * @param string $participantIdentifier
     * @param Pack $pack
     * @param Uuid $runId
     * @param string $actionBy
     * @param DateTimeInterface|null $timestamp
     * @param DateTimeInterface|null $simulation
     * @param Uuid|null $simulationId
     */
    public function __construct(
        string $studyCode,
        string $impCode,
        string $participantIdentifier,
        Pack $pack,
        Uuid $runId,
        string $actionBy,
        ?DateTimeInterface $timestamp = null,
        ?DateTimeInterface $simulation = null,
        ?Uuid $simulationId = null
    )
    {
        $this->study_code = $studyCode;
        $this->imp_code = $impCode;
        $this->participant_identifier = $participantIdentifier;
        $this->pack = $pack;
        $this->run_id = $runId;
        $this->allocated_by = $actionBy;
        $this->allocated = $timestamp ?? new DateTimeImmutable();
        $this->simulation = $simulation;
        $this->simulation_id = $simulationId;
    }

    //not in db
    #[AuditActor(AuditActionType::DELETE)]
    private(set) string $deleted_by;
    #[AuditTimestamp(AuditActionType::DELETE)]
    private(set) DateTimeInterface $deleted;
}
