<?php

declare(strict_types=1);

namespace Atlas\ImpBundle\Entity\Audit;

use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\ImpBundle\Exception\NotBlankException;
use Atlas\ImpBundle\Repository\Audit\AllocationRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Validator\Constraints\Length;

#[NotLogged]
#[ORM\Entity(repositoryClass: AllocationRepository::class)]
#[ORM\Index(columns: ['study_code', 'imp_code', 'participant_identifier', 'run_id', 'timestamp', 'simulation_id'])]
#[ORM\Table(name: 'audit_imp')]
class Imp
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    private(set) string $study_code;

    #[ORM\Column]
    private(set) string $imp_code;

    #[ORM\Column]
    private(set) string $participant_identifier;

    #[ORM\Column]
    private(set) string $message;

    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    private(set) string $action_by;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    private(set) DateTimeInterface $timestamp;

    #[ORM\Column(type: UuidType::NAME)]
    private(set) Uuid $run_id;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true, options: ['default' => null])]
    private(set) ?DateTimeInterface $simulation = null;

    #[ORM\Column(type: UuidType::NAME, nullable: true, options: ['default' => null])]
    private(set) ?Uuid $simulation_id = null;

    /**
     * @param string $studyCode
     * @param string $impCode
     * @param string $participantIdentifier
     * @param string $message
     * @param string $actionBy
     * @param Uuid $runId
     * @param ?DateTimeInterface $simulation
     * @param Uuid|null $simulationId
 */
    public function __construct(
        string $studyCode,
        string $impCode,
        string $participantIdentifier,
        string $message,
        string $actionBy,
        Uuid $runId,
        ?DateTimeInterface $simulation = null,
        ?Uuid $simulationId = null
    )
    {
        if(empty($studyCode = mb_trim($studyCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Study code');
        }

        if(empty($impCode = mb_trim($impCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Randomisation code');
        }

        if(empty($participantIdentifier = mb_trim($participantIdentifier, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Participant identifier');
        }

        if(empty($message = mb_trim($message, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Message');
        }

        if(empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Action by');
        }

        $this->study_code = $studyCode;
        $this->imp_code = $impCode;
        $this->participant_identifier = $participantIdentifier;
        $this->message = $message;
        $this->action_by = $actionBy;
        $this->timestamp = new DateTimeImmutable();
        $this->simulation = $simulation;
        $this->simulation_id = $simulationId;
        $this->run_id = $runId;
    }
}
