<?php

declare(strict_types=1);

namespace Atlas\ImpBundle\Entity\Imp;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\ImpBundle\Entity\Imp\Enum\PackLocation;
use Atlas\ImpBundle\Entity\Imp\Enum\PackStatus;
use Atlas\ImpBundle\Repository\Imp\PackRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;

#[ORM\Entity(repositoryClass: PackRepository::class)]
#[ORM\UniqueConstraint(columns: ['study_code', 'imp_code','pack_identifier'])]
#[ORM\Table(name: 'imp_pack')]
class Pack
{

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    public string $study_code;

    #[ORM\Column]
    private(set) string $imp_code;

    #[ORM\Column]
    private(set) string $pack_identifier;

    #[ORM\Column(nullable: true)]
    private(set) ?string $label;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true)]
    private(set) ?DateTimeInterface $pack_expiry = null;

    #[ORM\Column(nullable: true)]
    private(set) ?string $location_code = null;

    #[ORM\Column(type: Types::INTEGER, nullable: true, enumType: PackLocation::class)]
    private(set) ?PackLocation $location = null;

    #[ORM\Column(type: Types::INTEGER, nullable: true, enumType: PackStatus::class)]
    private(set) ?PackStatus $status = null;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $created;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $created_by;

    #[NotLogged]
    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::UPDATE)]
    private(set) DateTimeInterface $modified;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::UPDATE)]
    private(set) string $modified_by;

    #[NotLogged]
    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    private(set) string $reason;

    /**
     * @param string $studyCode
     * @param string $impCode
     * @param string $packIdentifier
     * @param string $actionBy
     * @param string|null $locationCode
     * @param string|null $label
     * @param DateTimeInterface|null $packExpiry
     * @param PackLocation|null $location
     * @param PackStatus|null $status
     */
    public function __construct(
        string $studyCode,
        string $impCode,
        string $packIdentifier,
        string $actionBy,
        ?string $locationCode = null,
        ?string $label = null,
        ?DateTimeInterface $packExpiry = null,
        ?PackLocation $location = null,
        ?PackStatus $status = null,
    ) {
        $this->study_code = $studyCode;
        $this->imp_code = $impCode;
        $this->pack_identifier = $packIdentifier;
        $this->label = $label;
        $this->pack_expiry = $packExpiry;
        $this->location_code = $locationCode;
        $this->status = $status;
        $this->location = $location;
        $this->created_by = $this->modified_by = $actionBy;
        $this->created = $this->modified = new DateTimeImmutable();
        $this->reason = 'Initial creation of pack';
    }

    public function setReceipt(PackStatus $status, string $actionBy): void
    {
        $this->status = $status;
        $this->location = PackLocation::LOCATION;

        $this->setModified($actionBy);
    }

    public function setOut(string $actionBy): void
    {
        $this->status = PackStatus::OUT;

        $this->setModified($actionBy);
    }

    public function setIn(string $actionBy): void
    {
        $this->status = PackStatus::IN;

        $this->setModified($actionBy);
    }

    public function setDispense(string $actionBy): void
    {
        $this->location = PackLocation::PARTICIPANT;

        $this->setModified($actionBy);
    }

    public function setDestroy(string $actionBy): void
    {
        $this->status = PackStatus::DESTROY;

        $this->setModified($actionBy);
    }

    public function setAllocate(string $actionBy): void
    {
        $this->status = PackStatus::ALLOCATE;

        $this->setModified($actionBy);
    }

    /**
     * @param string $actionBy
     * @return void
     */
    private function setModified(string $actionBy): void
    {
        $this->modified_by = $actionBy;
        $this->modified = new DateTimeImmutable();
    }
}
