<?php

namespace Atlas\ImpBundle\Service\Imp;

use Atlas\ImpBundle\Dto\LabelCountDto;
use Atlas\ImpBundle\Dto\PackCountDto;
use Atlas\ImpBundle\Entity\Imp\Enum\PackLocation;
use Atlas\ImpBundle\Entity\Imp\Enum\PackStatus;
use Atlas\ImpBundle\Entity\Imp\Pack;
use Atlas\ImpBundle\Repository\Imp\PackRepository;
use DateTimeInterface;
use Doctrine\ORM\EntityManagerInterface;

readonly class PackManager
{
    /**
     * @param EntityManagerInterface $entity_manager
     * @param PackRepository $packs
     */
    public function __construct(
        private EntityManagerInterface $entity_manager,
        private PackRepository $packs,
    )
    {
    }

    public function getPackFromPackIdentifier(string $packIdentifier): Pack
    {
        return $this->packs->findByPackIdentifierOrThrow($packIdentifier);
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @param bool $in
     * @return void
     */
    public function receipt(int|Pack $pack, string $actionBy, bool $in = true): void
    {
        if (is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setReceipt($in ? PackStatus::IN : PackStatus::OUT, $actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function in(int|Pack $pack, string $actionBy): void
    {
        if (is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setIn($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function out(int|Pack $pack, string $actionBy): void
    {
        if (is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setout($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function dispense(int|Pack $pack, string $actionBy): void
    {
        if (is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setDispense($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function return(int|Pack $pack, string $actionBy): void
    {
        if (is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setReturn($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param int|Pack $pack
     * @param string $actionBy
     * @return void
     */
    public function destroy(int|Pack $pack, string $actionBy): void
    {
        if (is_int($pack)) {
            $pack = $this->packs->findOrThrow($pack);
        }

        $pack->setDestroy($actionBy);
        $this->entity_manager->flush();
    }

    /**
     * @param string $studyCode
     * @param string $impCode
     * @param string $locationCode
     * @return PackCountDto
     */
    public function getLocationCounts(
        string $studyCode,
        string $impCode,
        string $locationCode
    ): PackCountDto
    {
        $result = $this->packs->generateLocationCounts($studyCode, $impCode, $locationCode);

        return new PackCountDto(
            $locationCode,
            $result['count'],
            $result['available'],
            $result['destroyed']
        );
    }

    /**
     * @param string $studyCode
     * @param string $impCode
     * @param string $locationCode
     * @param array $labels
     * @param PackLocation $location
     * @param PackStatus $status
     * @param bool $includeGlobal
     * @param bool $excludeExpired
     * @param DateTimeInterface|null $expiry
     * @return LabelCountDto[]
     */
    public function getLabelCounts(
        string $studyCode,
        string $impCode,
        string $locationCode,
        array $labels,
        PackLocation $location = PackLocation::LOCATION,
        PackStatus $status = PackStatus::IN,
        bool $includeGlobal = true,
        bool $excludeExpired = true,
        ?DateTimeInterface $expiry = null

    ): array
    {
        $result = $this->packs->generateLabelCountsForLocation(
            $studyCode,
            $impCode,
            $locationCode,
            $labels,
            location: $location,
            status: $status,
            includeGlobal: $includeGlobal,
            excludeExpired: $excludeExpired,
            expiry: $expiry
        );


        $counts = [];

        foreach ($result as $label => $count) {
            $counts[] = new LabelCountDto(
                label: $label,
                available: $count,
            );
        }

        return $counts;
    }

    public function isImpAvailable(
        string $studyCode,
        string $impCode,
        string $locationCode,
        array $labels,
        PackLocation $location = PackLocation::LOCATION,
        PackStatus $status = PackStatus::IN,
        bool $includeGlobal = true,
        bool $excludeExpired = true,
        ?DateTimeInterface $expiry = null
    ): bool
    {
        $result = $this->packs->generateLabelCountsForLocation(
            $studyCode,
            $impCode,
            $locationCode,
            $labels,
            location: $location,
            status: $status,
            includeGlobal: $includeGlobal,
            excludeExpired: $excludeExpired,
            expiry: $expiry
        );

        if($result === []) return false;

        return !in_array(0, $result, true);
    }
}
