<?php

declare(strict_types=1);

namespace Atlas\ImpBundle\Repository\Study;

use Atlas\ImpBundle\Entity\Study\Imp;
use Atlas\ImpBundle\Exception\ImpNotFoundException;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * IMP repository for study-scoped IMP entities.
 *
 * @extends ServiceEntityRepository<Imp>
 */
final class ImpRepository extends ServiceEntityRepository
{
    /**
     * Create repository.
     *
     * @param ManagerRegistry $registry Doctrine registry
     */
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Imp::class);
    }

    /**
     * Find an IMP by study_code + code or throw.
     *
     * @param string $studyCode Study code (e.g. EXC)
     * @param string $code IMP code (e.g. IMP1)
     * @return Imp
     * @throws ImpNotFoundException If no IMP exists for the given study/code
     */
    public function findByStudyImpCodeOrThrow(string $studyCode, string $code): Imp
    {
        $imp = $this->findByStudyImpCode($studyCode, $code);

        if ($imp === null) {
            throw ImpNotFoundException::fromCode($code);
        }

        return $imp;
    }

    /**
     * Find an IMP by study_code + code.
     *
     * @param string $studyCode Study code
     * @param string $code IMP code
     * @return ?Imp
     */
    public function findByStudyImpCode(string $studyCode, string $code): ?Imp
    {
        $studyCode = mb_trim($studyCode, encoding: 'UTF-8');
        $code = mb_trim($code, encoding: 'UTF-8');

        if ($studyCode === '' || $code === '') {
            return null;
        }

        return $this->createQueryBuilder('i')
            ->andWhere('i.study_code = :study_code')
            ->andWhere('i.code = :code')
            ->setParameter('study_code', mb_strtoupper($studyCode, encoding: 'UTF-8'))
            ->setParameter('code', mb_strtoupper($code, encoding: 'UTF-8'))
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }
}