<?php

namespace Atlas\ImpBundle\Repository\Imp;

use Atlas\ImpBundle\Entity\Imp\Allocation;
use Atlas\ImpBundle\Entity\Study\Imp;
use DateTimeImmutable;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Allocation>
 */
class AllocationRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Allocation::class);
    }

    /**
     * Return participant allocations as scalar rows.
     *
     * @param string $studyCode Study code
     * @param string $participantIdentifier Participant identifier
     * @return array<int, array{
     *     study_code: string,
     *     imp_code: string,
     *     imp_name: string,
     *     pack_label: string,
     *     pack_identifier: string,
     *     allocated: DateTimeImmutable,
     *     allocated_by: string
     * }>
     */
    public function findByParticipantAllocations(
        string $studyCode,
        string $participantIdentifier,
    ): array {
        /** @var array<int, array{
         *     study_code: string,
         *     imp_code: string,
         *     version: int,
         *     imp_name: string,
         *     pack_label: string,
         *     pack_identifier: string,
         *     allocated: DateTimeImmutable,
         *     allocated_by: string
         * }> $rows
         */
        $rows = $this->createQueryBuilder('a')
            ->innerJoin('a.pack', 'p')
            ->innerJoin(
                Imp::class,
                'i',
                'WITH',
                'i.study_code = a.study_code AND i.code = a.imp_code'
            )
            ->select([
                'a.study_code AS study_code',
                'a.imp_code AS imp_code',
                'a.version AS version',
                'i.name AS imp_name',
                'p.label AS pack_label',
                'p.pack_identifier AS pack_identifier',
                'a.allocated AS allocated',
                'a.allocated_by AS allocated_by',
            ])
            ->andWhere('a.study_code = :study_code')
            ->andWhere('a.participant_identifier = :participant_identifier')
            ->andWhere('a.simulation_id IS NULL')
            ->setParameter('study_code', $studyCode)
            ->setParameter('participant_identifier', $participantIdentifier)
            ->orderBy('a.allocated', 'DESC')
            ->getQuery()
            ->getArrayResult();

        return $rows;
    }

    /**
     * Return a participant allocation for a specific IMP as a single scalar row.
     *
     * @param string $studyCode Study code
     * @param string $participantIdentifier Participant identifier
     * @param string $impCode IMP code
     *
     * @return array{
     *     study_code: string,
     *     imp_code: string,
     *     version: int,
     *     imp_name: string,
     *     pack_label: string,
     *     pack_identifier: string,
     *     allocated: DateTimeImmutable,
     *     allocated_by: string
     * }|null
     *
     * @throws NonUniqueResultException If data integrity is broken (more than one row matches)
     */
    public function findParticipantAllocationForImp(
        string $studyCode,
        string $participantIdentifier,
        string $impCode,
    ): ?array {
        /** @var array{
         *     study_code: string,
         *     imp_code: string,
         *     version: int,
         *     imp_name: string,
         *     pack_label: string,
         *     pack_identifier: string,
         *     allocated: DateTimeImmutable,
         *     allocated_by: string
         * }|null $row
         */
        $row = $this->createQueryBuilder('a')
            ->innerJoin('a.pack', 'p')
            ->innerJoin(
                Imp::class,
                'i',
                'WITH',
                'i.study_code = a.study_code AND i.code = a.imp_code'
            )
            ->select([
                'a.study_code AS study_code',
                'a.imp_code AS imp_code',
                'a.version AS version',
                'i.name AS imp_name',
                'p.label AS pack_label',
                'p.pack_identifier AS pack_identifier',
                'a.allocated AS allocated',
                'a.allocated_by AS allocated_by',
            ])
            ->andWhere('a.study_code = :study_code')
            ->andWhere('a.participant_identifier = :participant_identifier')
            ->andWhere('a.imp_code = :imp_code')
            ->andWhere('a.simulation_id IS NULL')
            ->setParameter('study_code', $studyCode)
            ->setParameter('participant_identifier', $participantIdentifier)
            ->setParameter('imp_code', $impCode)
            ->getQuery()
            ->getOneOrNullResult(AbstractQuery::HYDRATE_ARRAY);

        return $row;
    }
}
