<?php
declare(strict_types=1);

namespace Atlas\ImpBundle\Tests\DependencyInjection;

use Atlas\ImpBundle\DependencyInjection\ImpExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

final class ImpExtensionTest extends TestCase
{
    protected function tearDown(): void
    {
        unset($_ENV['IMP_ENABLED'], $_SERVER['IMP_ENABLED']);
    }

    public function testLoadRegistersOnlyServicesAndRepositories(): void
    {
        $container = new ContainerBuilder();
        $container->setParameter('kernel.project_dir', \dirname(__DIR__, 2));

        $ext = new ImpExtension();
        $ext->load([], $container);

        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Service\\Imp\\PackManager'));
        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Service\\Specification\\SpecificationLoader'));
        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Service\\Specification\\SpecificationCache'));
        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Service\\Allocation\\Allocator'));
        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Service\\Allocation\\AllocationAuditor'));

        // repositories folder exists, so it must be registered
        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Repository\\Imp\\PackRepository'));
        self::assertTrue($container->hasDefinition('Atlas\\ImpBundle\\Repository\\Imp\\LevelRepository'));

        // never register Entities / DI as services
        self::assertFalse($container->hasDefinition('Atlas\\ImpBundle\\Entity\\Imp\\Pack'));
        self::assertFalse($container->hasDefinition('Atlas\\ImpBundle\\DependencyInjection\\ImpExtension'));
        self::assertFalse($container->hasDefinition('Atlas\\ImpBundle\\ImpBundle'));
    }

    public function testPrependDoesNothingWhenDisabled(): void
    {
        $container = new ContainerBuilder();
        $ext = new ImpExtension();

        $_ENV['IMP_ENABLED'] = '0';
        $_SERVER['IMP_ENABLED'] = '0';

        $ext->prepend($container);

        self::assertSame([], $container->getExtensionConfig('doctrine'));
    }

    public function testPrependAddsOrmMappingsWhenEnabledWithoutExplicitEms(): void
    {
        $container = new ContainerBuilder();
        $ext = new ImpExtension();

        $_ENV['IMP_ENABLED'] = '1';
        $_SERVER['IMP_ENABLED'] = '1';

        $ext->prepend($container);

        $doctrine = $container->getExtensionConfig('doctrine');
        self::assertNotEmpty($doctrine);

        $found = false;
        foreach ($doctrine as $cfg) {
            $m = $cfg['orm']['mappings']['AtlasImp'] ?? null;
            if (!is_array($m)) {
                continue;
            }
            self::assertSame('attribute', $m['type']);
            self::assertSame('Atlas\\ImpBundle\\Entity', $m['prefix']);
            self::assertFalse($m['is_bundle']);
            $found = true;
            break;
        }

        self::assertTrue($found);
    }

    public function testPrependAddsMappingsOnlyToDefaultEmWhenMultipleConfigured(): void
    {
        $container = new ContainerBuilder();
        $ext = new ImpExtension();

        $_ENV['IMP_ENABLED'] = '1';
        $_SERVER['IMP_ENABLED'] = '1';

        $container->prependExtensionConfig('doctrine', [
            'orm' => [
                'default_entity_manager' => 'phoenix',
                'entity_managers' => [
                    'phoenix' => [],
                    'legacy' => [],
                ],
            ],
        ]);

        $ext->prepend($container);

        $doctrine = $container->getExtensionConfig('doctrine');

        $foundPhoenix = false;
        $foundLegacy = false;

        foreach ($doctrine as $cfg) {
            $ems = $cfg['orm']['entity_managers'] ?? null;
            if (!is_array($ems)) {
                continue;
            }
            if (isset($ems['phoenix']['mappings']['AtlasImp'])) {
                $foundPhoenix = true;
            }
            if (isset($ems['legacy']['mappings']['AtlasImp'])) {
                $foundLegacy = true;
            }
        }

        self::assertTrue($foundPhoenix);
        self::assertFalse($foundLegacy);
    }
}