<?php

namespace Atlas\ImpBundle\Service\Imp;

use Atlas\ImpBundle\Dto\AllocationDto;
use Atlas\ImpBundle\Repository\Imp\AllocationRepository;
use Atlas\ImpBundle\Service\Specification\SpecificationLoader;
use DateMalformedStringException;
use DateTimeImmutable;
use Psr\Cache\InvalidArgumentException;

readonly class AllocationManager
{
    public function __construct(
        private AllocationRepository $allocations,
        private SpecificationLoader $specification_loader,
    ) {}

    /**
     * @param string $studyCode
     * @param string $participantIdentifier
     * @param bool $blinded
     * @return array<int, AllocationDto>
     * @throws InvalidArgumentException
     * @throws DateMalformedStringException
     */
    public function getParticipantAllocations(
        string $studyCode,
        string $participantIdentifier,
        bool $blinded = true
    ): array {
        $rows = $this->allocations->findByParticipantAllocations(
            $studyCode,
            $participantIdentifier
        );

        if ($rows === []) {
            return [];
        }

        $allocations = [];

        $spec_cache = [];

        foreach ($rows as $row) {

            $label = $row['pack_label']; // string by invariant

            $pack_label = null;

            if (!$blinded) {
                $imp_code = $row['imp_code'];
                $version = (int) $row['version'];

                $cache_key = $imp_code . '#v' . $version;

                $spec = $spec_cache[$cache_key]
                    ??= $this->specification_loader->load($studyCode, $imp_code, version: $version);

                $pack_label = $spec->label_map[$label] ?? $label;
            }

            $allocated = $row['allocated'];

            if (is_string($allocated)) {
                $allocated = new DateTimeImmutable($allocated);
            }

            $allocations[] = new AllocationDto(
                imp_code: $row['imp_code'],
                imp_name: $row['imp_name'],
                pack_identifier: $row['pack_identifier'],
                pack_label: $pack_label,
                allocated: $allocated,
                allocated_by: $row['allocated_by'],
            );
        }

        return $allocations;
    }
}