import { Controller } from '@hotwired/stimulus'

export default class extends Controller {
    static targets = ['container']
    static values = { scope: { type: String, default: 'global' } } // 'global' | 'local'

    connect() {
        this.onFlash = this.onFlash.bind(this)
        document.addEventListener('flash:add', this.onFlash)
    }

    disconnect() {
        document.removeEventListener('flash:add', this.onFlash)
    }

    onFlash(e) {
        const p = e.detail || {}
        const placement = (p.placement || 'global').toLowerCase()
        const klass = this.resolveClass(p)
        const msg = p.message || ''

        // Local controllers only accept flashes originating inside them
        if (this.scopeValue === 'local') {
            if (
                placement === 'local' &&
                (this.element === e.target || this.element.contains(e.target))
            ) {
                this.render(klass, msg)
            }
            return
        }

        // Global controller renders everything not explicitly local
        if (this.scopeValue === 'global' && placement !== 'local') {
            this.render(klass, msg)
        }
    }

    resolveClass(p) {
        if (p.bootstrap_class) return p.bootstrap_class.toLowerCase()
        const map = {
            success: 'success',
            error: 'danger',
            danger: 'danger',
            warning: 'warning',
            notice: 'info',
            info: 'info',
        }
        const t = (p.type || '').toLowerCase()
        return map[t] || 'secondary'
    }

    render(klass, message) {
        if (!this.hasContainerTarget) return

        const el = document.createElement('div')
        el.className = `alert alert-${klass} alert-dismissible fade show`
        el.setAttribute('role', 'alert')

        // textContent to avoid HTML injection
        el.appendChild(document.createTextNode(message))

        const btn = document.createElement('button')
        btn.type = 'button'
        btn.className = 'btn-close'
        btn.setAttribute('data-bs-dismiss', 'alert')
        btn.setAttribute('aria-label', 'Close')
        el.appendChild(btn)

        this.containerTarget.prepend(el)

        // Auto-dismiss after 6s
        setTimeout(() => {
            el.classList.remove('show')
            el.addEventListener('transitionend', () => el.remove(), { once: true })
        }, 6000)
    }
}
