import { Controller } from "@hotwired/stimulus";
import { Modal } from "bootstrap";

/**
 * Usage:
 *
 * <div id="reasonModal" data-controller="system-reason">
 *   <form method="post">
 *     <h5 data-system-reason-target="title"></h5>
 *     <label data-system-reason-target="label" for="reasonBox"></label>
 *     <textarea
 *       id="reasonBox"
 *       data-system-reason-target="textarea"
 *     ></textarea>
 *
 *     <div data-system-reason-target="counter"></div>
 *     <div data-system-reason-target="hiddenZone"></div>
 *
 *     <button type="submit" data-system-reason-target="submit"></button>
 *   </form>
 * </div>
 *
 * Trigger button example:
 *   data-bs-toggle="modal"
 *   data-bs-target="#reasonModal"
 *   data-system-reason-url="/thing/123/cancel"
 *   data-system-reason-title="Cancel booking?"
 *   data-system-reason-submit="Confirm cancel"
 *   data-system-reason-optional="false"
 *   data-system-reason-label="Why are you cancelling?"
 *   data-system-reason-fields='{"bookingId":"123","target":"#hidden-live-button"}'
 *
 * When submitted:
 *  - If a hidden field named "target" exists, we DON'T POST. Instead we:
 *      1. Take textarea value (trimmed)
 *      2. Put it into data-live-reason-param on that target button
 *      3. Click that button (so Symfony UX Live action runs)
 *      4. Hide modal
 *  - Otherwise we allow normal <form> submit (classic POST).
 */
export default class extends Controller {
    static targets = ["title", "label", "textarea", "counter", "hiddenZone", "submit"];

    connect() {
        this.handleShow = this.onShow.bind(this);
        this.handleSubmit = this.onSubmit.bind(this);
        this.handleInput = this.onInput.bind(this);

        this.element.addEventListener("show.bs.modal", this.handleShow);

        // we assume first <form> inside modal is "the" form
        this.formEl = this.element.querySelector("form");
        if (this.formEl) {
            this.formEl.addEventListener("submit", this.handleSubmit);
        }

        if (this.hasTextareaTarget) {
            this.textareaTarget.addEventListener("input", this.handleInput);
        }
    }

    disconnect() {
        this.element.removeEventListener("show.bs.modal", this.handleShow);

        if (this.formEl) {
            this.formEl.removeEventListener("submit", this.handleSubmit);
        }

        if (this.hasTextareaTarget) {
            this.textareaTarget.removeEventListener("input", this.handleInput);
        }
    }

    onShow(event) {
        const btn = event.relatedTarget;

        const url      = btn.getAttribute("data-system-reason-url")      || "#";
        const title    = btn.getAttribute("data-system-reason-title")    || "Confirm action";
        const submit   = btn.getAttribute("data-system-reason-submit")   || "Submit";
        const optional = (btn.getAttribute("data-system-reason-optional") || "false") === "true";
        const label    = btn.getAttribute("data-system-reason-label")    || "Reason";

        let fields = {};
        try {
            fields = JSON.parse(btn.getAttribute("data-system-reason-fields") || "{}");
        } catch {
            fields = {};
        }

        // UI
        this.titleTarget.textContent = title;
        this.submitTarget.textContent = submit;
        this.labelTarget.textContent = label;
        this.formEl.setAttribute("action", url);

        // Reset & apply constraints (10–255 unless optional)
        this.textareaTarget.value = "";
        this.textareaTarget.required = !optional;
        this.textareaTarget.setAttribute("minlength", optional ? "0" : "10");
        this.textareaTarget.setAttribute("maxlength", "255");

        this.updateCounter();

        // Hidden fields zone
        this.hiddenZoneTarget.innerHTML = "";
        Object.entries(fields).forEach(([name, value]) => {
            const input = document.createElement("input");
            input.type = "hidden";
            input.name = name;
            input.value = value ?? "";
            this.hiddenZoneTarget.appendChild(input);
        });
    }

    onInput() {
        this.updateCounter();
    }

    updateCounter() {
        const len = [...this.textareaTarget.value].length;
        this.counterTarget.textContent = `${len}/255`;
        this.counterTarget.classList.remove("text-danger");
    }

    onSubmit(e) {
        const optional = this.textareaTarget.required === false;
        const raw = this.textareaTarget.value.trim();
        const len = [...raw].length;

        // client-side length guard if required
        if (!optional && (len < 10 || len > 255)) {
            e.preventDefault();
            this.counterTarget.textContent = `Must be 10–255 chars (currently ${len}).`;
            this.counterTarget.classList.add("text-danger");
            this.textareaTarget.focus();
            return;
        }

        // "Live" handoff path
        const targetSel = this.element.querySelector('input[name="target"]')?.value;
        if (targetSel) {
            e.preventDefault(); // stop normal POST

            const btn = document.querySelector(targetSel);
            if (btn) {
                // write trimmed reason onto the Live action button
                btn.setAttribute('data-live-reason-param', raw);
                btn.click();
            }

            // close modal
            const modal = Modal.getOrCreateInstance(this.element);
            modal.hide();

            return;
        }

        // fallback normal submit, but send trimmed value
        this.textareaTarget.value = raw;
    }
}
