<?php

declare(strict_types=1);

namespace Modules\Paginator\Service;

use Doctrine\ORM\QueryBuilder;
use Doctrine\ORM\Tools\Pagination\Paginator as DoctrinePaginator;

final class Paginator
{
    private int $default_limit = 10;
    private int $max_limit = 100;

    /**
     * Paginate entities.
     *
     * @param QueryBuilder $queryBuilder
     * @param int $page
     * @param int|null $limit
     * @return array
     */
    public function paginate(QueryBuilder $queryBuilder, int $page = 1, ?int $limit = null): array
    {
        $limit = $limit ?? $this->default_limit;
        $limit = min($limit, $this->max_limit);
        $offset = ($page - 1) * $limit;

        // Set pagination in query
        $queryBuilder->setMaxResults($limit)
            ->setFirstResult($offset);

        // Get total results and paginated results
        $paginator = new DoctrinePaginator($queryBuilder);
        $totalResults = count($paginator);

        return [
            'items' => iterator_to_array($paginator),
            'total' => $totalResults,
            'page' => $page,
            'limit' => $limit,
            'pages' => (int)ceil($totalResults / $limit),
        ];
    }
}