<?php

declare(strict_types=1);

namespace Modules\Paginator\Service;

final readonly class PaginatorLinks
{
    /**
     * Generate pagination links.
     *
     * @param int $totalPages
     * @param int $currentPage
     * @param string $route
     * @param array $queryParams
     * @return string
     */
    public function generateLinks(int $totalPages, int $currentPage, string $route, array $queryParams = []): string
    {
        if ($totalPages <= 1) {
            return ''; // No need to display pagination if there's only one page.
        }

        $paginationHtml = '<nav aria-label="Page navigation"><ul class="pagination">';

        // Add "Previous" button
        $paginationHtml .= $this->generateLink(
            $currentPage - 1,
            'Previous',
            $route,
            $queryParams,
            activeClass: ($currentPage > 1) ? '' : 'disabled'
        );

        // Display pages 1-4
        for ($page = 1; $page <= min(2, $totalPages - 1); $page++) {
            $activeClass = ($page === $currentPage) ? 'active' : '';
            $paginationHtml .= $this->generateLink($page, (string) $page, $route, $queryParams, $activeClass);
        }

        // Add the current page and a couple of neighbors if we're past page 4
        $startPage = max(3, $currentPage);
        $endPage = min($currentPage + 2, $totalPages - 1);  // Stop one before the last page

        for ($page = $startPage; $page <= $endPage; $page++) {
            $activeClass = ($page === $currentPage) ? 'active' : '';
            $paginationHtml .= $this->generateLink($page, (string) $page, $route, $queryParams, $activeClass);
        }

        // Add the last page link
        $activeClass = ($totalPages === $currentPage) ? 'active' : '';
        $paginationHtml .= $this->generateLink($totalPages, (string) $totalPages, $route, $queryParams, $activeClass);

        // Add "Next" button
        $paginationHtml .= $this->generateLink(
            $currentPage + 1,
            'Next',
            $route,
            $queryParams,
            activeClass: ($currentPage < $totalPages) ? '' : 'disabled'
        );

        $paginationHtml .= '</ul></nav>';

        return $paginationHtml;
    }

    /**
     * Helper function to generate a pagination link.
     *
     * @param int $page
     * @param string $label
     * @param string $route
     * @param array $queryParams
     * @param string $activeClass
     * @return string
     */
    private function generateLink(int $page, string $label, string $route, array $queryParams, string $activeClass = ''): string
    {
        $queryParams['p'] = $page;
        $url = $route . '?' . http_build_query($queryParams);
        return sprintf('<li class="page-item %s"><a href="%s" class="page-link">%s</a></li>', $activeClass, $url, $label);
    }
}