<?php
declare(strict_types=1);

namespace Modules\PocockSimon\Domain;

use Atlas\RandomisationBundle\Domain\Specification as BaseSpecification;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use ValueError;

final readonly class Specification extends BaseSpecification
{
    /** @var Factor[] */
    private(set) array $factors;
    private(set) ImbalanceMetricEnum $imbalance_metric;
    private(set) float $prob_best;
    private(set) NoBestDistributionEnum $no_best_distribution;
    private(set) ?float $force_deterministic_if_gap;
    private(set) ?float $cap_prob_best;


    /**
     * @param string $system
     * @param string $code
     * @param array $arms
     * @param array $factors
     * @param string $imbalance_metric
     * @param float $prob_best
     * @param string $no_best_distribution
     * @param string|null $when
     * @param float|null $force_deterministic_if_gap
     * @param string|null $weighting
     * @param string|null $setVariable
     * @param int|null $simpleForFirst
     * @param float|null $cap_prob_best
     * @param array $forceValues
     * @param array $pid
     * @param string|null $apiToken
     * @param array $requiredVariables
     * @param string|null $actionBy
     */
    public function __construct(
        string $system,
        string $code,
        array $arms,
        array $factors,
        string $imbalance_metric,
        float $prob_best,
        string $no_best_distribution,
        ?string $when = null,
        ?float $force_deterministic_if_gap = null,
        ?string $weighting = null,
        ?string $setVariable = null,
        ?int $simpleForFirst = null,
        ?float $cap_prob_best = null,
        array $forceValues = [],
        array $pid = [],
        ?string $apiToken = null,
        array $requiredVariables = [],
        ?string $actionBy = null,
    )
    {

        $factorArray = [];
        $seen =[];

        foreach($factors as $key => $factor) {
            $f = new Factor(
                $factor['name'] ?? '',
                $key,
                $factor['labels'] ?? [],
                $factor['weight'] ?? 1.0,
                $factor['from'] ?? null
            );

            if(isset($seen[$f->name])) {
                throw new SpecificationException(sprintf('factors[%s].name "%s" is a duplicate.', $key, $f->name));
            }

            $seen[$f->name] = true;
            $factorArray[] = $f;
        }

        $this->factors = $factorArray;

        if(count($this->factors) === 0) {
            throw new SpecificationException('There must be at least one factor');
        }

        try {
            $this->imbalance_metric = ImbalanceMetricEnum::from($imbalance_metric);
        }
        catch (ValueError) {
            throw new SpecificationException('imbalance.imbalance_metric is invalid');
        }

        if ($prob_best <= 0 || $prob_best >= 1) {
            throw new SpecificationException('imbalance.prob_best must be in the range (0,1)');
        }
        $this->prob_best = $prob_best;

        try {
            $this->no_best_distribution = NoBestDistributionEnum::from($no_best_distribution);
        } catch (ValueError) {
            throw new SpecificationException('imbalance.nonbest_distribution is invalid');
        }

        if($force_deterministic_if_gap !== null && $force_deterministic_if_gap <= 0) {
            throw new SpecificationException('imbalance.force_deterministic_if_gap must be more than 0 if set');
        }

        $this->force_deterministic_if_gap = $force_deterministic_if_gap;

        if ($cap_prob_best !== null) {
            if ($cap_prob_best <= 0 || $cap_prob_best > 1) {
                throw new SpecificationException('safeguards.cap_prob_best must be greater than 0 and ≤ 1');
            }
        }

        $this->cap_prob_best = $cap_prob_best;

        parent::__construct(
            $system,
            $code,
            $arms,
            when: $when,
            weighting: $weighting,
            setVariable: $setVariable,
            simpleForFirst: $simpleForFirst,
            forceValues: $forceValues,
            pid: $pid,
            apiToken: $apiToken,
            requiredVariables: $requiredVariables,
            actionBy: $actionBy
        );
    }

    public function getType(): string
    {
        return 'pocock_simon';
    }
}
