<?php

namespace Modules\PocockSimon\Entity;

use Atlas\AuditBundle\Attribute\NotLogged;
use Atlas\RandomisationBundle\Exception\NotBlankException;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Validator\Constraints\NotBlank;

#[NotLogged]
#[ORM\Entity]
#[ORM\Table(
    name: 'rand_ps_total',
    indexes: [
        new ORM\Index(columns: ['study_code']),
        new ORM\Index(columns: ['randomisation_code']),
        new ORM\Index(columns: ['participant_identifier']),
        new ORM\Index(columns: ['arm']),
        new ORM\Index(columns: ['run_id']),
        new ORM\Index(columns: ['simulation_id'])
    ])
]
#[ORM\UniqueConstraint(columns: ['study_code','randomisation_code','participant_identifier','arm','simulation_id'])]
class Total
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    public string $study_code;

    #[ORM\Column]
    private(set) string $randomisation_code;

    #[ORM\Column]
    private(set) string $participant_identifier;

    #[ORM\Column]
    public string $arm;

    #[ORM\Column]
    public string $total;

    #[ORM\Column(type: UuidType::NAME)]
    private(set) ?Uuid $run_id = null;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    private(set) DateTimeInterface $randomised;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    private(set) string $randomised_by;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE, nullable: true, options: ['default' => null])]
    private(set) ?DateTimeInterface $simulation = null;

    #[ORM\Column(type: UuidType::NAME, nullable: true, options: ['default' => null])]
    private(set) ?Uuid $simulation_id = null;

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param string $participantIdentifier
     * @param string $arm
     * @param int $total
     * @param Uuid $runId
     * @param string $actionBy
     * @param DateTimeInterface|null $timestamp
     * @param DateTimeInterface|null $simulation
     * @param Uuid|null $simulationId
     */
    public function __construct(
        string $studyCode,
        string $randomisationCode,
        string $participantIdentifier,
        string $arm,
        int $total,
        Uuid $runId,
        string $actionBy,
        ?DateTimeInterface $timestamp = null,
        ?DateTimeInterface $simulation = null,
        ?Uuid $simulationId = null
    )
    {
        if(empty($studyCode = mb_trim($studyCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Study code');
        }

        if(empty($randomisationCode = mb_trim($randomisationCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Randomisation code');
        }

        if(empty($participantIdentifier = mb_trim($participantIdentifier, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Participant identifier');
        }

        if(empty($arm = mb_trim($arm, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Arm');
        }

        if(empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Action by');
        }

        $this->study_code = $studyCode;
        $this->randomisation_code = $randomisationCode;
        $this->participant_identifier = $participantIdentifier;
        $this->arm = $arm;
        $this->total = $total;
        $this->randomised_by = $actionBy;
        $this->run_id = $runId;
        $this->randomised = $timestamp ?? new DateTimeImmutable();
        $this->simulation = $simulation;
        $this->simulation_id = $simulationId;
    }

}
