<?php
declare(strict_types=1);

namespace Modules\PocockSimon\Tests\DependencyInjection;

use Modules\PocockSimon\DependencyInjection\PocockSimonExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

final class PocockSimonExtensionTest extends TestCase
{
    protected function tearDown(): void
    {
        unset($_ENV['SHARED_RANDOMISATION_ENABLED'], $_ENV['SHARED_RANDOMISATION_ALGS']);
        unset($_SERVER['SHARED_RANDOMISATION_ENABLED'], $_SERVER['SHARED_RANDOMISATION_ALGS']);
    }

    public function testLoadRegistersOnlyServiceAndRepositoryNamespaces(): void
    {
        $container = new ContainerBuilder();

        // Critical: many Extensions rely on this for locating config files.
        $container->setParameter('kernel.project_dir', \dirname(__DIR__, 2));

        $ext = new PocockSimonExtension();
        $ext->load([], $container);

        // Assert expected DI services exist (only if the class exists).
        $this->assertServiceIfClassExists($container, 'Modules\\PocockSimon\\Service\\Algorithm');
        $this->assertServiceIfClassExists($container, 'Modules\\PocockSimon\\Service\\Parser');
        $this->assertServiceIfClassExists($container, 'Modules\\PocockSimon\\Service\\Exporter');
        $this->assertServiceIfClassExists($container, 'Modules\\PocockSimon\\Service\\Reporter');

        // Assert repositories are only registered if present.
        $this->assertServiceIfClassExists($container, 'Modules\\PocockSimon\\Repository\\HypotheticalRepository');
        $this->assertServiceIfClassExists($container, 'Modules\\PocockSimon\\Repository\\TotalRepository');

        // Never registered as services (regardless of existence).
        self::assertFalse($container->hasDefinition('Modules\\PocockSimon\\Entity\\Total'));
        self::assertFalse($container->hasDefinition('Modules\\PocockSimon\\DependencyInjection\\PocockSimonExtension'));
        self::assertFalse($container->hasDefinition('Modules\\PocockSimon\\PocockSimonBundle'));

        // Also ensure we didn't accidentally register whole src/ via wildcards
        // (guards against regressions in services.yaml).
        foreach (\array_keys($container->getDefinitions()) as $id) {
            self::assertFalse(\str_starts_with($id, 'Modules\\PocockSimon\\Entity\\'), 'Entity classes must not be registered as services.');
            self::assertFalse(\str_starts_with($id, 'Modules\\PocockSimon\\DependencyInjection\\'), 'DI classes must not be registered as services.');
        }
    }

    public function testPrependDoesNothingWhenAlgorithmIsNotEnabled(): void
    {
        $container = new ContainerBuilder();
        $ext = new PocockSimonExtension();

        $_ENV['SHARED_RANDOMISATION_ENABLED'] = '0';
        $_ENV['SHARED_RANDOMISATION_ALGS'] = '["pocock_simon"]';
        $_SERVER['SHARED_RANDOMISATION_ENABLED'] = '0';
        $_SERVER['SHARED_RANDOMISATION_ALGS'] = '["pocock_simon"]';

        $ext->prepend($container);

        self::assertSame([], $container->getExtensionConfig('doctrine'));
    }

    public function testPrependAddsDoctrineMappingsWithoutExplicitEntityManagers(): void
    {
        $container = new ContainerBuilder();
        $ext = new PocockSimonExtension();

        $_ENV['SHARED_RANDOMISATION_ENABLED'] = '1';
        $_ENV['SHARED_RANDOMISATION_ALGS'] = '["pocock_simon"]';
        $_SERVER['SHARED_RANDOMISATION_ENABLED'] = '1';
        $_SERVER['SHARED_RANDOMISATION_ALGS'] = '["pocock_simon"]';

        $ext->prepend($container);

        $doctrine = $container->getExtensionConfig('doctrine');
        self::assertNotEmpty($doctrine);

        $found = false;
        foreach ($doctrine as $cfg) {
            $m = $cfg['orm']['mappings']['PocockSimon'] ?? null;
            if (!\is_array($m)) {
                continue;
            }

            self::assertSame('attribute', $m['type']);
            self::assertSame('Modules\\PocockSimon\\Entity', $m['prefix']);
            self::assertFalse($m['is_bundle']);

            $found = true;
            break;
        }

        self::assertTrue($found, 'Expected PocockSimon mapping to be prepended under orm.mappings.');
    }

    public function testPrependAddsDoctrineMappingsOnlyToDefaultEntityManagerWhenMultipleAreConfigured(): void
    {
        $container = new ContainerBuilder();
        $ext = new PocockSimonExtension();

        $_ENV['SHARED_RANDOMISATION_ENABLED'] = '1';
        $_ENV['SHARED_RANDOMISATION_ALGS'] = '["pocock_simon"]';
        $_SERVER['SHARED_RANDOMISATION_ENABLED'] = '1';
        $_SERVER['SHARED_RANDOMISATION_ALGS'] = '["pocock_simon"]';

        // Existing doctrine config with explicit EMs
        $container->prependExtensionConfig('doctrine', [
            'orm' => [
                'default_entity_manager' => 'phoenix',
                'entity_managers' => [
                    'phoenix' => [],
                    'legacy' => [],
                ],
            ],
        ]);

        $ext->prepend($container);

        $doctrine = $container->getExtensionConfig('doctrine');

        $foundPhoenix = false;
        $foundLegacy = false;

        foreach ($doctrine as $cfg) {
            $ems = $cfg['orm']['entity_managers'] ?? null;
            if (!\is_array($ems)) {
                continue;
            }

            if (isset($ems['phoenix']['mappings']['PocockSimon'])) {
                $foundPhoenix = true;
            }
            if (isset($ems['legacy']['mappings']['PocockSimon'])) {
                $foundLegacy = true;
            }
        }

        self::assertTrue($foundPhoenix, 'Expected mapping to be attached to the configured default EM (phoenix).');
        self::assertFalse($foundLegacy, 'Mapping must not be added to non-default EMs.');
    }

    private function assertServiceIfClassExists(ContainerBuilder $container, string $fqcn): void
    {
        if (!\class_exists($fqcn)) {
            // Avoid false negatives if the class/folder isn't present in this module.
            self::markTestSkipped(sprintf('Class %s does not exist in this module.', $fqcn));
        }

        self::assertTrue(
            $container->hasDefinition($fqcn),
            sprintf('%s should be registered as a service.', $fqcn)
        );
    }
}