<?php
declare(strict_types=1);

namespace Modules\PocockSimon\Service;

use Atlas\RandomisationBundle\Contract\ReportInterface;
use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Repository\Participant\FactorRepository;
use Atlas\ReportRenderBundle\Model\Chart as RenderChart;
use Modules\PocockSimon\Domain\Factor as PsFactor;
use Symfony\Component\DependencyInjection\Attribute\AutoconfigureTag;
use Symfony\Component\Uid\Uuid;
use Symfony\UX\Chartjs\Builder\ChartBuilderInterface;
use Symfony\UX\Chartjs\Model\Chart as ChartJs;

#[AutoconfigureTag('atlas.randomisation.reporter', attributes: ['algorithm' => 'pocock_simon'])]
final readonly class Reporter implements ReportInterface
{
    public function __construct(
        private ChartBuilderInterface $charts,
        private FactorRepository $factors
    ) {}

    public function getType(): string
    {
        return 'pocock_simon';
    }

    /**
     * One pie per (factor, raw value).
     * - Slices = arms; values = counts
     * - Arm order = spec order (fallback: alpha by arm)
     * - Title shows label if available, else raw value
     */
    public function buildAlgorithmSections(
        string $studyCode,
        string $randomisationCode,
        ?SpecificationInterface $specification = null,
        ?Uuid $simulated = null
    ): array {
        // rows: ['factor' => F, 'level' => L(raw), 'arm' => A, 'n' => int]
        $rows = $this->factors->countsByFactorLevelArm(
            $studyCode,
            $randomisationCode,
            simulateId: $simulated
        );

        if (!$rows) {
            return [];
        }

        // Arm order: prefer spec order, else discovered alpha
        $arms = [];
        if ($specification && \is_iterable($specification->arms)) {
            foreach ($specification->arms as $armName => $_) {
                $arms[] = (string)$armName;
            }
        } else {
            $armSet = [];
            foreach ($rows as $r) {
                $armSet[(string)$r['arm']] = true;
            }
            $arms = array_keys($armSet);
            sort($arms, \SORT_STRING);
        }

        // Factor label maps (raw => pretty), keyed by factor name
        $labelMaps = [];
        if ($specification && property_exists($specification, 'factors')) {
            foreach ($specification->factors as $f) {
                /** @var PsFactor $f */
                $labelMaps[$f->name] = $f->labels ?? [];
            }
        }

        // Group counts: $by[factor][raw][arm] = n
        $by = [];
        foreach ($rows as $r) {
            $f = (string)$r['factor'];
            $l = (string)$r['level']; // raw value as stored
            $a = (string)$r['arm'];
            $n = (int)$r['n'];
            $by[$f][$l][$a] = ($by[$f][$l][$a] ?? 0) + $n;
        }

        $sections = [];
        $palette  = RenderChart::getColours(max(1, count($arms)));

        ksort($by, \SORT_STRING);
        foreach ($by as $factor => $levels) {
            ksort($levels, \SORT_STRING);

            foreach ($levels as $rawLevel => $armCounts) {
                // Align to $arms; fill missing with 0
                $data  = [];
                $total = 0;
                foreach ($arms as $arm) {
                    $v = (int)($armCounts[$arm] ?? 0);
                    $data[] = $v;
                    $total += $v;
                }
                if ($total === 0) {
                    continue; // skip truly empty pies
                }

                $pretty = $labelMaps[$factor][$rawLevel] ?? (string)$rawLevel;
                $title  = sprintf('%s: %s', $factor, $pretty);

                $pie = $this->charts->createChart(ChartJs::TYPE_PIE);
                $pie->setData([
                    'labels'   => $arms,
                    'datasets' => [[
                        'label'           => $title,
                        'data'            => $data,
                        'backgroundColor' => $palette,
                    ]],
                ]);
                $pie->setOptions([
                    'plugins' => [
                        'legend'  => ['position' => 'bottom'],
                        'tooltip' => ['enabled' => true],
                    ],
                    'responsive' => true,
                ]);

                // 3-up grid on xl, 2-up on md, 1-up on xs
                $sections[] = new RenderChart(
                    title: $title,
                    chart: $pie,
                    columns: 'col-12 col-md-6 col-xl-4'
                );
            }
        }

        return $sections; // RenderChart[]
    }
}
