<?php

declare(strict_types=1);

namespace Modules\PocockSimon\Domain;

use Atlas\RandomisationBundle\Exception\SpecificationException;

final readonly class Factor
{
    private(set) string $name;
    private(set) float $weight;

    /** @var array<int|string,string> */
    private(set) array $labels;
    private(set) ?string $from;

    /**
     * @param string $name
     * @param int $factorKey
     * @param array $labels
     * @param float $weight
     * @param string|null $from
     */
    public function __construct(
        string $name,
        int $factorKey,
        array $labels = [],
        float $weight = 1.0,
        ?string $from = null
    )
    {
        if (empty($name = mb_trim($name, encoding: 'UTF-8'))) {
            throw new SpecificationException(sprintf('factors[%s].name is required', $factorKey));
        }

        $labelsArray = [];

        foreach ($labels as $key => $label) {
            $label = mb_trim($label, encoding: 'UTF-8');

            if ($label === '') {
                throw new SpecificationException(sprintf('factors[%s].labels[%s] cannot be empty.', $factorKey, $key));
            }

            if(array_key_exists($key, $labelsArray)) {
                throw new SpecificationException(sprintf('factors[%s].labels[%s] "%s" is a duplicate.', $factorKey, $key, $label));
            }

            $labelsArray[$key] = $label;
        }

        if($weight <= 0) throw new SpecificationException(sprintf('factors[%s].weight must be > 0.', $factorKey));

        if($from !== null) {
            $from = mb_trim($from, encoding: 'UTF-8');
            if($from === '') {
                throw new SpecificationException(sprintf('factors[%s].from cannot be empty if set', $factorKey));
            }
        }

        $this->name = $name;
        $this->labels = $labelsArray;
        $this->weight = $weight;
        $this->from = $from;
    }
}