<?php

declare(strict_types=1);

namespace Modules\PocockSimon\Service;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Contract\SpecificationParserInterface;
use Atlas\RandomisationBundle\Service\Specification\Traits\KeyCheckerTrait;
use Modules\PocockSimon\Domain\Specification;
use Symfony\Component\DependencyInjection\Attribute\AutoconfigureTag;

#[AutoconfigureTag('atlas.randomisation.parser', attributes: ['algorithm' => 'pocock_simon'])]
class Parser implements SpecificationParserInterface
{

    use KeyCheckerTrait;

    private const array EXCLUDE = [
        'imbalance' => [
            'tie_breaker', //always the same in PS so we don't get from spec
            'randomisation_probability', // only used in AB tie_breaker = random
            'bias'
        ]
    ];

    public function getType(): string
    {
        return 'pocock_simon';
    }

    /**
     * Parse a "pocock and simon" randomisation specification array into a domain object.
     *
     * @param array $specification Raw specification array loaded from disk.
     *
     * @return SpecificationInterface Parsed, validated specification for "pocock and simon" randomisation.
     */
    public function parse(array $specification): SpecificationInterface
    {
        //other algorithms might require additional parsing

        $this->assertBadKeys($specification, self::EXCLUDE);

        //otherwise it is relly simple, the domain model will call everything else out!
        return new Specification(
            $specification['system'],
            $specification['code'],
            $specification['randomisation']['arms'],
            $specification['factors'],
            $specification['imbalance']['imbalance_metric'],
            $specification['imbalance']['prob_best'],
            $specification['imbalance']['nonbest_distribution'],
            $specification['imbalance']['force_deterministic_if_gap'] ?? null,
            $specification['randomisation']['arm_weighting'] ?? null,
            $specification['set_variable'] ?? null,
            $specification['safeguards']['first_n_complete_random'] ?? null,
            $specification['safeguards']['cap_prob_best'] ?? null,
            $specification['force_values'] ?? [],
            $specification['pid'] ?? [],
            $specification['api_token'] ?? null,
            $specification['required_variables'] ?? [],
            $specification['action_by'] ?? null
        );
    }
}