<?php
declare(strict_types=1);

namespace Modules\PocockSimon\DependencyInjection;

use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Extension\PrependExtensionInterface;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\DependencyInjection\Extension\Extension;
use function dirname;

final class PocockSimonExtension extends Extension implements PrependExtensionInterface
{
    public function load(array $configs, ContainerBuilder $container): void
    {
        $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../../config'));
        $loader->load('services.yaml');
    }

    public function prepend(ContainerBuilder $container): void
    {
        if (!$this->isEnabled($container)) {
            return;
        }

        $doctrineConfigs = $container->getExtensionConfig('doctrine');

        $defaultEm = 'default';
        $entityManagers = [];

        foreach ($doctrineConfigs as $config) {
            if (isset($config['orm']['default_entity_manager'])) {
                $defaultEm = $config['orm']['default_entity_manager'];
            }

            if (isset($config['orm']['entity_managers']) && is_array($config['orm']['entity_managers'])) {
                // later configs override earlier
                $entityManagers = array_replace($entityManagers, $config['orm']['entity_managers']);
            }
        }

        if (is_string($defaultEm)) {
            $defaultEm = trim($defaultEm);
        }

        $mappings = [
            'PocockSimon' => [
                'is_bundle' => false,
                'type' => 'attribute',
                'dir' => dirname(__DIR__) . '/Entity',
                'prefix' => 'Modules\\PocockSimon\\Entity',
            ],
        ];

        if (!isset($entityManagers[$defaultEm])) {
            // No explicit EMs defined -> old/simple projects
            $container->prependExtensionConfig('doctrine', [
                'orm' => [
                    'mappings' => $mappings,
                ],
            ]);
        } else {
            // Multi-EM project -> attach ONLY to the *configured default* EM (phoenix here)
            $container->prependExtensionConfig('doctrine', [
                'orm' => [
                    'entity_managers' => [
                        $defaultEm => [
                            'mappings' => $mappings,
                        ],
                    ],
                ],
            ]);
        }
    }

    private function isEnabled(ContainerBuilder $container): bool
    {
        $raw = $container->resolveEnvPlaceholders('%env(json:SHARED_RANDOMISATION_ALGS)%', true);

        if (!$raw) {
            return false;
        }

        if (is_string($raw)) {
            $list = json_decode($raw, true);
        } else {
            $list = $raw;
        }

        if (!is_array($list)) {
            return false;
        }

        return in_array('pocock_simon', $list, true);
    }
}
