<?php
declare(strict_types=1);

namespace Modules\PocockSimon\Service;

use Atlas\ExportBundle\Service\Exporter as FileExporter;
use Atlas\RandomisationBundle\Contract\ExportInterface;
use Atlas\RandomisationBundle\Dto\ExportTypeDto;
use Modules\PocockSimon\Repository\HypotheticalRepository;
use Modules\PocockSimon\Repository\TotalRepository;
use Symfony\Component\DependencyInjection\Attribute\AutoconfigureTag;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Uid\Uuid;

#[AutoconfigureTag('atlas.randomisation.exporter', attributes: ['algorithm' => 'pocock_simon'])]
final readonly class Exporter implements ExportInterface
{
    public function __construct(
        private HypotheticalRepository $hypotheticals,
        private TotalRepository $totals,
        private FileExporter $files
    ) {}

    public function getType(): string { return 'pocock_simon'; }

    public function getAvailableExports(): array
    {
        return [
            new ExportTypeDto('Pocock–Simon Hypotheticals (pivoted)', 'hypotheticals'),
            new ExportTypeDto('Pocock–Simon Totals (pivoted)',        'totals'),
        ];
    }

    public function export(
        string $studyCode,
        string $randomisationCode,
        string $exportType,
        string $method = self::CSV,
        ?Uuid $simulated = null
    ): Response {
        return match ($exportType) {
            'hypotheticals' => $this->exportHypotheticals($studyCode, $randomisationCode, $method, $simulated),
            'totals'        => $this->exportTotals($studyCode, $randomisationCode, $method, $simulated),
            default         => $this->files->csv([['error' => 'Unknown PS export type']]),
        };
    }

    // ---------- HYPOTHETICALS (pivot: Arm × Factor × Value × Metric) ----------

    private function exportHypotheticals(
        string $studyCode,
        string $randomisationCode,
        string $method,
        ?Uuid $simulated
    ): Response {
        $rows = $this->hypotheticals->exportRows($studyCode, $randomisationCode, $simulated);

        // Discover arms & factor→values from observed data
        $armSet  = [];
        $factors = []; // factor => [values...]
        foreach ($rows as $h) {
            $armSet[(string)$h['arm']] = true;
            $f = (string)$h['factor'];
            $v = (string)$h['value'];
            $factors[$f] ??= [];
            if ($v !== '' && !in_array($v, $factors[$f], true)) {
                $factors[$f][] = $v;
            }
        }

        $arms = array_keys($armSet);
        sort($arms, SORT_NATURAL | SORT_FLAG_CASE);
        ksort($factors, SORT_NATURAL | SORT_FLAG_CASE);
        foreach ($factors as &$vals) sort($vals, SORT_NATURAL | SORT_FLAG_CASE);
        unset($vals);

        // Header
        $header = ['participant_identifier', 'run_id'];
        foreach ($arms as $arm) {
            foreach ($factors as $factor => $values) {
                foreach ($values as $value) {
                    $base = sprintf('%s — %s=%s', $arm, $factor, $value);
                    $header[] = $base.' — count';
                    $header[] = $base.' — hypothetical';
                    $header[] = $base.' — factor_score';
                }
            }
        }

        if (!$rows) {
            return $this->emit($method, $header, []);
        }

        // Index: [$pid][$run][$arm][$factor][$value] => metrics
        $byKey = [];
        foreach ($rows as $h) {
            $pid = (string)$h['participant_identifier'];
            $run = (string)$h['run_id'];
            $arm = (string)$h['arm'];
            $fac = (string)$h['factor'];
            $val = (string)$h['value'];
            $byKey[$pid][$run][$arm][$fac][$val] = [
                'count'        => (int)$h['count'],
                'hypothetical' => (int)$h['hypothetical'],
                'factor_score' => (int)$h['factor_score'],
            ];
        }

        // Rows
        $matrix = [];
        foreach ($byKey as $pid => $runs) {
            foreach ($runs as $run => $cube) {
                $row = [
                    'participant_identifier' => $pid,
                    'run_id'                 => $run,
                ];
                foreach ($arms as $arm) {
                    foreach ($factors as $factor => $values) {
                        foreach ($values as $value) {
                            $cell = $cube[$arm][$factor][$value] ?? null;
                            $row[sprintf('%s — %s=%s — count',        $arm, $factor, $value)] = $cell['count']        ?? 0;
                            $row[sprintf('%s — %s=%s — hypothetical', $arm, $factor, $value)] = $cell['hypothetical'] ?? 0;
                            $row[sprintf('%s — %s=%s — factor_score', $arm, $factor, $value)] = $cell['factor_score'] ?? 0;
                        }
                    }
                }
                $matrix[] = array_replace(array_fill_keys($header, null), $row);
            }
        }

        return $this->emit($method, $header, $matrix);
    }

    // ---------- TOTALS (pivot: Arm × total) ----------

    private function exportTotals(
        string $studyCode,
        string $randomisationCode,
        string $method,
        ?Uuid $simulated
    ): Response {
        $rows = $this->totals->exportRows($studyCode, $randomisationCode, $simulated);

        $armSet = [];
        foreach ($rows as $t) $armSet[(string)$t['arm']] = true;
        $arms = array_keys($armSet);
        sort($arms, SORT_NATURAL | SORT_FLAG_CASE);

        $header = array_merge(
            ['participant_identifier', 'run_id'],
            array_map(fn(string $a) => sprintf('%s — total', $a), $arms)
        );

        if (!$rows) {
            return $this->emit($method, $header, []);
        }

        // group per participant+run
        $byKey = []; // [$pid][$run][$arm] = total
        foreach ($rows as $t) {
            $pid = (string)$t['participant_identifier'];
            $run = (string)$t['run_id'];
            $arm = (string)$t['arm'];
            $byKey[$pid][$run][$arm] = (int)$t['total'];
        }

        $matrix = [];
        foreach ($byKey as $pid => $runs) {
            foreach ($runs as $run => $armTotals) {
                $row = [
                    'participant_identifier' => $pid,
                    'run_id'                 => $run,
                ];
                foreach ($arms as $arm) {
                    $row[sprintf('%s — total', $arm)] = $armTotals[$arm] ?? 0;
                }
                $matrix[] = array_replace(array_fill_keys($header, null), $row);
            }
        }

        return $this->emit($method, $header, $matrix);
    }

    // ---------- Emit helpers ----------

    private function emit(string $method, array $header, array $matrix): Response
    {
        $rows = $this->asSequential($header, $matrix);

        return match ($method) {
            self::CSV   => $this->files->csv($rows),
            self::EXCEL => $this->files->excel($rows),
            self::SPSS  => $this->files->spss($rows),
            default     => $this->files->csv($rows),
        };
    }

    /** Convert associative rows to the exact column order expected by the file writers. */
    private function asSequential(array $header, array $rows): array
    {
        $out = [];
        foreach ($rows as $r) {
            $line = [];
            foreach ($header as $h) {
                // keep associative keys so the exporter can emit a single header
                $line[$h] = $r[$h] ?? null;
            }
            $out[] = $line;
        }
        return $out;
    }
}
