<?php
declare(strict_types=1);

namespace Modules\Simple\Service;

use Atlas\RandomisationBundle\Contract\AlgorithmInterface;
use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Dto\RandomisationResultDto;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Repository\Randomisation\InactiveArmRepository;
use Atlas\RandomisationBundle\Repository\Randomisation\InactiveRandomisationRepository;
use Atlas\RandomisationBundle\Service\Randomisation\RandomisationAuditor;
use DateTimeInterface;
use Modules\Simple\Domain\Specification;
use Random\RandomException;
use Symfony\Component\DependencyInjection\Attribute\AutoconfigureTag;
use Symfony\Component\Uid\Uuid;

#[AutoconfigureTag('atlas.randomisation.algorithm', attributes: ['algorithm' => 'simple'])]
final readonly class Algorithm implements AlgorithmInterface
{

    public function __construct(
        private RandomisationAuditor $logger,
        private InactiveRandomisationRepository $inactive_randomisation,
        private InactiveArmRepository $inactive_arms,
        private AllocationRepository $allocations
    )
    {
    }

    public function getType(): string
    {
        return 'simple';
    }

    /**
     * @param string $studyCode
     * @param string $specificationName
     * @param Specification $specification
     * @param string $participantIdentifier
     * @param string $location
     * @param string $actionBy
     * @param array $variables
     * @param DateTimeInterface|null $simulate
     * @param Uuid|null $simulateId
     * @return RandomisationResultDto|false
     * @throws RandomException
     */
    public function randomise(
        string $studyCode,
        string $specificationName,
        SpecificationInterface $specification,
        string $participantIdentifier,
        string $location,
        string $actionBy,
        array $variables = [],
        ?DateTimeInterface $simulate = null,
        ?Uuid $simulateId = null
    ): RandomisationResultDto|false
    {

        $studyCode = mb_strtoupper($studyCode, encoding: 'utf-8');
        $randomisationCode = mb_strtoupper($specification->code, encoding: 'utf-8');

        $uri = Uuid::v7();

        $this->logger->log($studyCode, $randomisationCode , $participantIdentifier, sprintf('Randomisation started using algorithm %s', $this->getType()), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
        //1: check that the randomisation is viable

        $inactive = $this->inactive_randomisation->check($studyCode, $randomisationCode, $location);

        if ($inactive) {
            $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, 'Randomisation inactive for all or current locations', $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);

            return false;
        }

        //2: check arms
        $activeArms = [];

        foreach ($specification->arms as $name => $arm) {

            $inactive = $this->inactive_arms->check($studyCode, $randomisationCode, $name, $location);

            if ($inactive) {
                $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm %s inactive (removed)', $name), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
            } else {
                $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm %s active', $name), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
                $activeArms[$name] = $arm->weight;
            }
        }

        if (count($activeArms) < 1) {
            $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, 'No available arms', $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
            return false;
        }
        if (count($activeArms) === 1) {
            $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, 'Only 1 arm available and allocated', $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
            return new RandomisationResultDto(array_key_first($activeArms), $uri);
        }

        //3: now check for safeguarding
        if ($specification->simple_for_first !== null) {

            $simpleForFirstCount = $this->allocations->countInitialRandomAllocations($studyCode, $randomisationCode, simulationId: $simulateId);

            if($simpleForFirstCount < $specification->simple_for_first) {
                $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, 'Safeguarding: pure simple randomisation', $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
                $arm = array_rand($activeArms);
                return new RandomisationResultDto($arm, $uri, ['type' => sprintf('initial-random{%s}', $simpleForFirstCount + 1)]);
            }
        }

        //4: Otherwise simple randomisation based on weights
        foreach ($activeArms as $arm => $weight) {
            $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm %s weighting is %s', $arm, $weight), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
        }

        $total = array_sum($activeArms);

        $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Weighted total is %s', $total), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);

        $rand = random_int(1, $total);

        $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Random draw is %s', $rand), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);

        $cumulative = 0;

        $allocated = null;

        foreach ($activeArms as $arm => $weight) {

            $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm %s cumulative %s + weight %s = %s', $arm, $cumulative, $weight, $cumulative + $weight), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);

            $cumulative += $weight;
            if ($rand <= $cumulative) {
                $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm %s cumulative %s >= random draw %s', $arm, $cumulative, $rand), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);

                $allocated = $arm;
                break;
            }
            else {
                $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm %s cumulative %s < random draw %s', $arm, $cumulative, $rand), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
            }
        }

        //5: might be a big problem so fail if null
        if ($allocated === null) {
            $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, 'Allocation failed (no arm matched cumulative). Aborting.', $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);
            return false;
        }

        //6: Allocate the arm
        $this->logger->log($studyCode, $randomisationCode, $participantIdentifier, sprintf('Arm allocated %s', $allocated), $actionBy, $uri, simulate: $simulate, simulateId: $simulateId);

        return new RandomisationResultDto($allocated, $uri, [ 'type' => 'algorithmic' ]);
    }
}
