<?php

declare(strict_types=1);

namespace Modules\Simple\Service;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Contract\SpecificationParserInterface;
use Atlas\RandomisationBundle\Service\Specification\Traits\KeyCheckerTrait;
use Modules\Simple\Domain\Specification;
use Symfony\Component\DependencyInjection\Attribute\AutoconfigureTag;

#[AutoconfigureTag('atlas.randomisation.parser', attributes: ['algorithm' => 'simple'])]
class Parser implements SpecificationParserInterface
{

    use KeyCheckerTrait;

    private const array EXCLUDE = [
        'factors',
        'imbalance',
        'safeguards' => [
            'cap_prob_best'
        ],
    ];

    public function getType(): string
    {
        return 'simple';
    }

    /**
     * Parse a "simple" randomisation specification array into a domain object.
     *
     * Expected structure (top-level keys):
     *  - code (string)                     Unique code / form code.
     *  - system (string)                   Target system ('redcap', 'macro', 'phoenix', etc.).
     *  - api_token (string)                API token (required for REDCap/MACRO, forbidden for Phoenix).
     *  - required_variables (string[])     Variables that must be present before randomisation.
     *  - when (string)                     Boolean expression gate for whether randomisation is allowed.
     *  - set_variable (string|null)        Destination field to write the allocated arm (REDCap/MACRO only).
     *  - force_values (array[])            Key/value pairs to set after randomisation.
     *  - pid (array)                       Optional PID config:
     *      - generate (bool)
     *      - format (string)
     *      - set_variable (string)
     *  - action_by (string|null)           Field containing the username / site user performing the action.
     *  - safeguards (array)                Optional safeguards:
     *      - first_n_complete_random (int) Number of initial participants to allocate purely at random.
     *  - randomisation (array)             Algorithm config:
     *      - type (string)                 Must be "simple" for this parser.
     *      - arms (string[])               List of arm labels.
     *      - arm_weighting (string|null)   Ratio string like "2:2:1" matching the number of arms.
     *
     * Behaviour:
     *  - Asserts that disallowed keys for simple randomisation are not present
     *    (e.g. stratification / minimisation settings).
     *  - Returns a SimpleSpecification value object. The SimpleSpecification
     *    constructor will enforce per-field validation (system rules, weighting
     *    shape, PID rules, etc.) and throw SpecificationException on failure.
     *
     * @param array $specification Raw specification array loaded from disk.
     *
     * @return SpecificationInterface Parsed, validated specification for "simple" randomisation.
     */
    public function parse(array $specification): SpecificationInterface
    {
        //other algorithms might require additional parsing

        $this->assertBadKeys($specification, self::EXCLUDE);

        //otherwise it is relly simple, the domain model will call everything else out!
        return new Specification(
            $specification['system'] ?? '',
            $specification['code'] ?? '',
            $specification['randomisation']['arms'] ?? [],
            when: $specification['when'] ?? '',
            weighting:$specification['randomisation']['arm_weighting'] ?? null,
            setVariable: $specification['set_variable'] ?? null,
            simpleForFirst: $specification['safeguards']['first_n_complete_random'] ?? null,
            forceValues: $specification['force_values'] ?? [],
            pid: $specification['pid'] ?? [],
            apiToken: $specification['api_token'] ?? null,
            requiredVariables: $specification['required_variables'] ?? [],
            actionBy: $specification['action_by'] ?? null
        );
    }
}