<?php

declare(strict_types=1);

namespace Atlas\PdfGeneratorBundle\Service;

use RuntimeException;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;

final readonly class PdfGenerator
{
    public function __construct(
        private ParameterBagInterface $params
    ) {}

    public function generateFromHtml(string $html): string
    {
        if (trim($html) === '') {
            throw new RuntimeException('Refusing to render empty HTML into PDF');
        }

        // project root (bundle installed under /vendor/atlas/randomisation-bundle/…)
        $projectDir = $this->params->get('kernel.project_dir');
        $binary = $projectDir . '/vendor/h4cc/wkhtmltopdf-amd64/bin/wkhtmltopdf-amd64';

        if (!is_file($binary) || !is_executable($binary)) {
            throw new RuntimeException("wkhtmltopdf binary not found or not executable at $binary");
        }

        $tmpHtml = tempnam(sys_get_temp_dir(), 'atlas_pdf_');
        $tmpPdf  = tempnam(sys_get_temp_dir(), 'atlas_pdf_');
        if (!$tmpHtml || !$tmpPdf) {
            throw new RuntimeException('Failed to create temp files for PDF generation');
        }

        $tmpHtml .= '.html';
        $tmpPdf  .= '.pdf';
        file_put_contents($tmpHtml, $html);

        $cmd = [
            $binary,
            '--disable-local-file-access',
            '--disable-javascript',
            '--margin-top', '10',
            '--margin-right', '10',
            '--margin-bottom', '10',
            '--margin-left', '10',
            '--lowquality',
            $tmpHtml,
            $tmpPdf,
        ];

        $process = new Process($cmd, null, [], null, 60);
        $process->run();

        if (!$process->isSuccessful()) {
            @unlink($tmpHtml);
            @unlink($tmpPdf);
            throw new ProcessFailedException($process);
        }

        $pdf = file_get_contents($tmpPdf);
        @unlink($tmpHtml);
        @unlink($tmpPdf);

        if ($pdf === false) {
            throw new RuntimeException('wkhtmltopdf produced no output');
        }

        return $pdf;
    }
}
