<?php

namespace Atlas\RandomisationBundle\Controller;

use Atlas\RandomisationBundle\Exception\ReportException;
use Atlas\RandomisationBundle\Service\Report\ReportBuilder;
use Atlas\ReportRenderBundle\Service\ReportRenderer;
use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Uid\Uuid;

#[IsGranted('ROLE_USER')]
final class ReportController extends AbstractController
{
    public function __construct(
        #[Autowire(param: 'shared.randomisation.reports_code')]
        private readonly string $report_permission,
        #[Autowire(param: 'shared.randomisation.blinded_code')]
        private readonly string $blinded_permission,
        #[Autowire(param: 'shared.randomisation.permission_needs_study')]
        private readonly bool $permission_needs_study = true
    )
    {
    }

    /**
     * @param Request $request
     * @param string $study
     * @param string $randomisation
     * @param string $specification
     * @param ReportBuilder $reporter
     * @param ReportRenderer $renderer
     * @return Response
     * @throws InvalidArgumentException
     */
    #[Route(
        '/report/{study<[A-Za-z0-9_-]+>}/{randomisation<[A-Za-z0-9_-]+>}/{specification<[A-Za-z0-9_-]+>}',
        name: 'app_rand_report',
        methods: ['GET'])]
    public function report(
        Request $request,
        string $study,
        string $randomisation,
        string $specification,
        ReportBuilder $reporter,
        ReportRenderer $renderer
    ): Response
    {

        $this->checkPermissions($study);

        $simulated = $request->query->get('simulated');

        if($simulated) {
            if (!Uuid::isValid($simulated)) {
                throw new ReportException('Invalid simulation id');
            } else {
                $simulated = Uuid::fromString($simulated);
            }
        }

        $layout = $reporter->generate($study, $randomisation, $specification, simulated: $simulated);

        return $renderer->renderHtmlResponse($layout);

    }

    /**
     * @param string $study
     * @return void
     */
    public function checkPermissions(string $study): void
    {
        if ($this->permission_needs_study) {
            $this->denyAccessUnlessGranted(
                attribute: 'permission',
                subject: [
                    'permission' => $this->report_permission,
                    'study' => $study
                ]
            );

            $this->getBlinded($study);

            return;
        }

        $this->denyAccessUnlessGranted(
            attribute: 'permission',
            subject: [
                'permission' => $this->report_permission
            ]
        );

        $this->getBlinded();
    }

    /**
     * @param string|null $study
     * @return void
     */
    private function getBlinded(?string $study = null): void
    {
        $blinded = $study !== null
            ?
            $this->isGranted(
                attribute: 'permission',
                subject: [
                    'permission' => $this->blinded_permission,
                    'study' => $study
                ]
            )
            :
            $this->isGranted(
                attribute: 'permission',
                subject: [
                    'permission' => $this->blinded_permission,
                ]
            );

        if($blinded) throw new AccessDeniedHttpException('Blinded users cannot see reports.');
    }
}
