<?php

namespace Atlas\RandomisationBundle\Controller;


use Atlas\RandomisationBundle\Exception\RandomisationException;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use Atlas\RandomisationBundle\Service\Simulator\SimulationRunner;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use Atlas\RandomisationBundle\Service\Spreadsheet\ImporterService as SpreadsheetImporter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Cache\Exception\CacheException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Throwable;

#[IsGranted('ROLE_USER')]
final class SimulateController extends AbstractController
{


    private const array EXT  = ['xlsx', 'xls', 'csv'];
    private const array MIME  = [
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'application/vnd.ms-excel',
        'text/csv',
        'application/csv',
        'text/plain',
        'application/octet-stream',
    ];

    public function __construct(
        #[Autowire(param: 'shared.randomisation.simulate_code')]
        private readonly string $permission,
        #[Autowire(param: 'shared.randomisation.permission_needs_study')]
        private readonly bool $permission_needs_study = true
    )
    {}

    #[Route('/simulate/{code<[A-Za-z0-9_-]+>}', name: 'app_rand_simulate', methods: ['GET'])]
    public function index(
        string $code
    ): Response {

        $this->checkPermissions($code);

        return $this->render('@Randomisation/simulate/index.html.twig', [
            'study' => $code
        ]);
    }

    #[Route('/simulate/{code<[A-Za-z0-9_-]+>}', name: 'app_rand_simulate_upload', methods: ['POST'])]
    public function upload(
        string $code,
        Request $request,
        SpecificationLoader $loader,
        SpreadsheetImporter $importer,
        SimulationRunner $runner
    ): Response {

        $this->checkPermissions($code);

        if (!$this->isCsrfTokenValid('simulate_upload_'.$code, (string)$request->request->get('_token'))) {
            return $this->errorRedirect('Invalid session token.', 'app_rand_simulate', ['code' => $code]);
        }

        $file = $request->files->get('spreadsheet');

        if (!$file instanceof UploadedFile) {
            return $this->errorRedirect(
                'No file uploaded.',
                'app_rand_simulate',     // fixed name
                ['code' => $code]
            );
        }

        $ext  = mb_strtolower($file->getClientOriginalExtension(), encoding: 'UTF-8');
        $mime = (string) $file->getMimeType();

        if (!in_array($ext, self::EXT, true)) {
            return $this->errorRedirect('Unsupported file type.', 'app_rand_simulate', ['code' => $code]);
        }
        if (!in_array($mime, self::MIME, true)) {
            return $this->errorRedirect('Unsupported MIME type.', 'app_rand_simulate', ['code' => $code]);
        }

        if ($file->getSize() !== null && $file->getSize() > 20 * 1024 * 1024) {
            return $this->errorRedirect('File too large (max 20 MB).', 'app_rand_simulate', ['code' => $code]);
        }

        $randomisation = trim((string) $request->request->get('randomisation', ''));
        $version = $request->request->getInt('version', -1);

        if (empty($randomisation) || $version < 0) {
            return $this->errorRedirect(
                'Randomisation code/name is required and version must be a non-negative integer.',
                'app_rand_simulate',
                ['code' => $code]
            );
        }

        $user = $this->getUser();

        try {
            // Load specification
            $specification = $loader->load($code, $randomisation, $version);

            // Import spreadsheet rows
            $rows = $importer->import($file);

            // Run simulation
            $runner->run($code, $randomisation, $version, $rows, $user->getUserIdentifier(), specification: $specification);

        } catch (SpecificationException $e) {
            return $this->errorRedirect($e->getMessage(), 'app_rand_simulate', ['code' => $code]);

        } catch (CacheException) {
            return $this->errorRedirect(
                'Cannot access specification from cache.',
                'app_rand_simulate',
                ['code' => $code]
            );

        } catch (RandomisationException $e) {

            $this->handleRandomisationFailure($e);

            return $this->redirectToRoute('app_rand_simulate', ['code' => $code]);

        } /*catch (Throwable $e) {

            return $this->errorRedirect(sprintf('An unexpected error occurred: %s', $e->getMessage()), 'app_rand_simulate', ['code' => $code]);
        }*/

        $this->addFlash('success', 'File processed successfully! You can now download it.');
        return $this->redirectToRoute('app_rand_simulate', ['code' => $code]);
    }

    /**
     * @param string $message
     * @param string $route
     * @param array $params
     * @return Response
     */
    private function errorRedirect(string $message, string $route, array $params = []): Response
    {
        $this->addFlash('error', $message);
        return $this->redirectToRoute($route, $params);
    }

    /**
     * @param RandomisationException $e
     * @return void
     */
    private function handleRandomisationFailure(RandomisationException $e): void
    {

        // Detailed error messages for each failed test
        $results = method_exists($e, 'getResults') ? $e->getResults() : [];

        if (!empty($results)) {
            foreach ($results as $failure) {
                $reason = $failure['reason'];
                $message = sprintf(
                    'Test "%s" failed: %s',
                    $failure['test'],
                    $reason instanceof \Throwable ? $reason->getMessage() : (string)$reason
                );
                $this->addFlash('error', $message);
            }
        } else {
            $this->addFlash('error', $e->getMessage());
        }
    }

    /**
     * @param string $code
     * @return void
     */
    public function checkPermissions(string $code): void
    {
        if ($this->permission_needs_study) {
            $this->denyAccessUnlessGranted(
                attribute: 'permission',
                subject: [
                    'permission' => $this->permission,
                    'study' => $code
                ]
            );

            return;
        }

        $this->denyAccessUnlessGranted(
            attribute: 'permission',
            subject: [
                'permission' => $this->permission
            ]
        );
    }

}
