<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Repository\Randomisation;

use Atlas\RandomisationBundle\Dto\AllocationDto;
use Atlas\RandomisationBundle\Entity\Randomisation\Allocation;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;

/**
 * @extends ServiceEntityRepository<Allocation>
 */
class AllocationRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Allocation::class);
    }

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param Uuid|null $simulated
     * @return AllocationDto[]
     */
    public function getAllocations(
        string $studyCode,
        string $randomisationCode,
        ?Uuid $simulated = null
    ): array
    {
        $qb = $this->createQueryBuilder('a');

        $qb
            ->select(
                "NEW Atlas\\RandomisationBundle\\Dto\\AllocationDto(
                a.run_id,
                a.participant_identifier,
                a.randomisation_group,
                a.type,
                a.arm_allocated,
                a.randomised_on,
                a.randomised_by
            )"
            )
            ->andWhere('a.study_code = :studyCode')
            ->setParameter('studyCode', $studyCode)
            ->andWhere('a.randomisation_code = :randomisationCode')
            ->setParameter('randomisationCode', $randomisationCode);

        if ($simulated !== null) {
            $qb->andWhere('a.simulation_id = :simulationId')
                ->setParameter('simulationId', $simulated, UuidType::NAME);
        } else {
            $qb->andWhere('a.simulation_id IS NULL');
        }

        return $qb->getQuery()->getResult();
    }

    /**
     * Counts the number of allocations per arm for a given randomisation.
     *
     * The logic handles the optional simulation ID:
     * - If $simulationId is provided, it counts only for that specific simulation.
     * - If $simulationId is null, it counts only for real-world (non-simulated) randomisation.
     *
     * @param string $studyCode
     * @param string $randomisationCode
     * @param Uuid|null $simulationId The ID of the simulation run to count, or null for real runs.
     * @return array<string, int> An associative array where the key is the arm name and the value is the count.
     */
    public function countAllocationsByArm(
        string $studyCode,
        string $randomisationCode,
        ?Uuid $simulationId = null
    ): array
    {
        $qb = $this->createQueryBuilder('a');

        // 1. SELECT the arm and COUNT the records, grouping by arm
        $qb->select('a.arm_allocated as arm', 'COUNT(a.id) as count')
            ->groupBy('a.arm_allocated');

        // 2. Always filter by study and randomisation code
        $qb->andWhere('a.study_code = :studyCode')
            ->setParameter('studyCode', $studyCode);

        $qb->andWhere('a.randomisation_code = :randomisationCode')
            ->setParameter('randomisationCode', $randomisationCode);

        // 3. Conditional Filtering based on $simulationId (Core Logic)
        if ($simulationId !== null) {
            // If ID is provided, count only for that specific simulation ID
            $qb->andWhere('a.simulation_id = :simulationId')
                ->setParameter('simulationId', $simulationId, UuidType::NAME);
        } else {
            // If ID is NULL, count only for non-simulated/real-world runs (where simulation_id IS NULL)
            $qb->andWhere('a.simulation_id IS NULL');
        }

        $results = $qb->getQuery()->getArrayResult();

        // 4. Transform the result array into the desired [arm_name => count] format
        $counts = [];
        foreach ($results as $row) {
            $counts[$row['arm']] = (int) $row['count'];
        }

        return $counts;
    }

    public function getSimulationMapForStudy(string $studyCode): array
    {
        // pull DISTINCT pairs (randomisation_code, simulation_id)
        $rows = $this->createQueryBuilder('a')
            ->select('DISTINCT a.randomisation_code as randomisation_code, a.simulation_id as simulation_id')
            ->where('a.study_code = :studyCode')
            ->andWhere('a.simulation_id IS NOT NULL')
            ->setParameter('studyCode', $studyCode)
            ->getQuery()
            ->getArrayResult();

        // build map like ['RANDOMISATION_A' => ['uuid1', 'uuid2'], ...]
        $map = [];

        foreach ($rows as $row) {
            $randCode = $row['randomisation_code'];
            $simId = $row['simulation_id']->toRfc4122();

            if (!isset($map[$randCode])) {
                $map[$randCode] = [];
            }

            $map[$randCode][] = $simId;
        }

        // ensure uniqueness + clean index
        foreach ($map as $randCode => $simIds) {
            $map[$randCode] = array_values(array_unique($simIds));
        }

        return $map;
    }

    public function getSimulationIdsForRandomisation(string $studyCode, string $randomisationCode): array
    {
        $rows = $this->createQueryBuilder('a')
            ->select('DISTINCT a.simulation_id AS simulation_id')
            ->where('a.study_code = :studyCode')
            ->andWhere('a.randomisation_code = :randCode')
            ->andWhere('a.simulation_id IS NOT NULL')
            ->setParameter('studyCode', $studyCode)
            ->setParameter('randCode', $randomisationCode)
            ->getQuery()
            ->getArrayResult();

        $ids = [];
        foreach ($rows as $row) {
            // now we expect simulation_id to be a Uuid object again
            $ids[] = $row['simulation_id']->toRfc4122();
        }

        return array_values(array_unique($ids));
    }

    /**
     * Counts all allocations of type "initial-random-{x}" for a given study and randomisation.
     *
     * @param string $studyCode
     * @param string $randomisationCode
     * @param Uuid|null $simulationId
     * @return int
     */
    public function countInitialRandomAllocations(string $studyCode, string $randomisationCode, ?Uuid $simulationId = null): int
    {
        $qb = $this->createQueryBuilder('a')
            ->select('COUNT(a.id)')
            ->where('a.study_code = :study')
            ->andWhere('a.randomisation_code = :rand')
            ->andWhere('a.type LIKE :type')
            ->setParameter('study', $studyCode)
            ->setParameter('rand', $randomisationCode)
            ->setParameter('type', 'initial-random%');
        if ($simulationId !== null) {
            // we're in a simulated run → only count previous rows from this same simulation
            $qb->andWhere('a.simulation_id = :sim')
                ->setParameter('sim', $simulationId, UuidType::NAME);
        } else {
            // we're in a real run → only count real rows
            $qb->andWhere('a.simulation_id IS NULL');
        }

        return (int) $qb->getQuery()->getSingleScalarResult();
    }
}
