<?php

namespace Atlas\RandomisationBundle\Controller;

use Atlas\RandomisationBundle\Exception\ReportException;
use Atlas\RandomisationBundle\Repository\Randomisation\AllocationRepository;
use Atlas\RandomisationBundle\Security\RandomisationAuthorisationInterface;
use Atlas\RandomisationBundle\Service\Export\ExportGenerator;
use Psr\Cache\InvalidArgumentException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Uid\Uuid;

#[IsGranted('ROLE_USER')]
final readonly class ExportController extends AbstractController
{
    public function __construct(
        private readonly RandomisationAuthorisationInterface $authorisation
    )
    {
    }

    /**
     * @throws InvalidArgumentException
     */
    #[Route(
        '/export/randomisation/{study<[A-Za-z0-9_-]+>}',
        name: 'shared_rand_export',
        defaults: ['_uppercase' => ['study']],
        methods: ['GET'])
    ]
    public function index(
        string $study,
        ExportGenerator $exporter,
        AllocationRepository $allocations,
    ): Response
    {

        if (!$this->authorisation->canExport($study)) {
            throw new AccessDeniedHttpException();
        }

        if ($this->authorisation->isBlinded($study)) {
            throw new AccessDeniedHttpException();
        }

        $exports = $exporter->list($study);

        $merged = array();

        foreach ($exports as $key => $items) {
            $merged[$key]['algorithms'] = $items;
            $merged[$key]['simulations'] = $allocations->getSimulationIdsForRandomisation($study, $key);
        }

        return $this->render('@Randomisation/export/index.html.twig', [
            'study' => $study,
            'exports' => $merged
        ]);
    }

    /**
     * @param Request $request
     * @param string $study
     * @param string $location
     * @param string $code
     * @param string $algorithm
     * @param string $type
     * @param string $method
     * @param ExportGenerator $exporter
     * @return Response
     */
    #[Route(
        '/export/randomisation/{code<[A-Za-z0-9_-]+>}/algorithm/{algorithm<[A-Za-z0-9_-]+>}/study/{study<[A-Za-z0-9_-]+>}/{type<[A-Za-z0-9_-]+>}/{method<[A-Za-z0-9_-]+>}',
        name: 'shared_rand_export_file',
        defaults: ['_uppercase' => ['code', 'study']],
        methods: ['GET'])
    ]
    public function export(
        Request $request,
        string $study,
        string $code,
        string $algorithm,
        string $type,
        string $method,
        ExportGenerator $exporter
    ): Response
    {

        if (!$this->authorisation->canExport($study)) {
            throw new AccessDeniedHttpException();
        }

        if ($this->authorisation->isBlinded($study)) {
            throw new AccessDeniedHttpException();
        }

        $simulated = $request->query->get('simulated');

        if ($simulated) {
            if (!Uuid::isValid($simulated)) {
                throw new ReportException('Invalid simulation id');
            }
            $simulated = Uuid::fromString($simulated);
        }

        return $exporter->export($study, $code, $algorithm, $type, $method, simulated: $simulated);
    }
}
