<?php

namespace Atlas\RandomisationBundle\Entity\Randomisation;

use Atlas\AuditBundle\Attribute\AuditActor;
use Atlas\AuditBundle\Attribute\AuditTimestamp;
use Atlas\AuditBundle\Attribute\Enum\AuditActionType;
use Atlas\RandomisationBundle\Exception\NotBlankException;
use Atlas\RandomisationBundle\Repository\Randomisation\InactiveArmRepository;
use DateTimeImmutable;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints\Length;
use Symfony\Component\Validator\Constraints\NotBlank;

#[ORM\Entity(repositoryClass: InactiveArmRepository::class)]
#[ORM\Table(
    name: 'rando_inactive_randomisation',
    indexes: [
        new ORM\Index(columns: ['study_code']),
        new ORM\Index(columns: ['randomisation_code']),
        new ORM\Index(columns: ['location']),
        new ORM\Index(columns: ['arm'])
    ]
)]

#[ORM\UniqueConstraint(columns: ['study_code', 'randomisation_code', 'location', 'arm'])]
#[UniqueEntity(fields: ['study_code', 'randomisation_code', 'location', 'arm'])]
class InactiveRandomisation
{

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private(set) ?int $id = null;

    #[ORM\Column]
    private(set) string $study_code;

    #[ORM\Column]
    private(set) string $randomisation_code;

    #[ORM\Column]
    private(set) string $location; //if ALL globally inactive

    #[ORM\Column]
    private(set) string $arm;

    #[ORM\Column(type: Types::DATETIME_IMMUTABLE)]
    #[AuditTimestamp(AuditActionType::INSERT)]
    private(set) DateTimeInterface $added;

    #[ORM\Column(length: 255)]
    #[NotBlank]
    #[AuditActor(AuditActionType::INSERT)]
    private(set) string $added_by;

    #[ORM\Column(length: 255)]
    #[Length(min: 10, max: 255)]
    #[NotBlank]
    private(set) string $reason;

    //none db fields used for listener
    #[AuditTimestamp(AuditActionType::DELETE)]
    private(set) DateTimeInterface $deleted;
    #[AuditActor(AuditActionType::DELETE)]
    private(set) string $deleted_by;

    /**
     * @param string $studyCode
     * @param string $randomisationCode
     * @param string $actionBy
     * @param string $reason
     * @param string|null $location
     */
    public function __construct(
        string $studyCode,
        string $randomisationCode,
        string $actionBy,
        string $reason,
        ?string $location = null

    ) {

        if(empty($studyCode = mb_trim($studyCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Study code');
        }

        if(empty($randomisationCode = mb_trim($randomisationCode, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Log code');
        }

        if($location === null || empty($location = mb_trim($location, encoding: 'UTF-8'))) {
            $location = 'ALL';
        }

        if(empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Reason');
        }

        if(empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Action by');
        }

        $this->study_code = $studyCode;
        $this->randomisation_code = $randomisationCode;
        $this->location = $location;
        $this->added_by = $actionBy;
        $this->added = new DateTimeImmutable();
        $this->reason = $reason;
    }

    /**
     * @param string $reason
     * @param string $actionBy
     * @return void
     * @throws NotBlankException
     */
    public function preDelete(string $reason, string $actionBy): void
    {

        if(empty($reason = mb_trim($reason, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Reason');
        }

        if(empty($actionBy = mb_trim($actionBy, encoding: 'UTF-8'))) {
            throw NotBlankException::name('Action by');
        }

        $this->deleted_by = $actionBy;
        $this->deleted = new DateTimeImmutable();
        $this->reason = $reason;
    }

    /**
     * @var bool
     */
    public bool $is_global {
        get =>  $this->location === 'ALL';
    }

}