<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Domain;

use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Domain\Enum\SystemEnum;
use Atlas\RandomisationBundle\Domain\Arm;
use Atlas\RandomisationBundle\Exception\SpecificationException;
use ValueError;

readonly abstract class Specification implements SpecificationInterface
{
    private(set) ?string $when;
    private(set) SystemEnum $system;
    private(set) string $code;
    /** @var array<string, Arm> */
    private(set) array $arms;
    private(set) ?string $set_variable;
    private(set) ?int $simple_for_first;
    /** @var array<string, ForceValue> */
    private(set) array $force_values;
    private(set) bool $pid_generate;
    private(set) ?string $pid_format;
    private(set) ?string $pid_set_variable;
    private(set) ?string $api_token;
    /** @var string[] */
    private(set) array $required_variables;
    private(set) ?string $action_by;

    /**
     * @param string $system
     * @param string $code
     * @param array $arms
     * @param string|null $when
     * @param string|null $weighting
     * @param string|null $setVariable
     * @param int|null $simpleForFirst
     * @param array $forceValues
     * @param array $pid
     * @param string|null $apiToken
     * @param string[] $requiredVariables
     * @param string|null $actionBy
     */
    public function __construct(
        string $system,
        string $code,
        array $arms,
        ?string $when = null,
        ?string $weighting = null,
        ?string $setVariable = null, //not needed for Phoenix
        ?int $simpleForFirst = null,
        array $forceValues = [],
        array $pid = [],
        ?string $apiToken = null,
        array $requiredVariables = [],
        ?string $actionBy = null,
    )
    {
        if($when != null) {
            $when = mb_trim($when, encoding: 'UTF-8');
            if($when === '') $this->when = null;
            else $this->when = $when;
        }

        try {
            $this->system = SystemEnum::from($system);
        }
        catch (ValueError) {
            throw new SpecificationException('System does not exist');
        }

        if (empty($code = mb_trim($code, encoding: 'UTF-8'))) {
            throw new SpecificationException('randomisation.code is required.');
        }

        $this->code = $code;

        $weighting = mb_trim($weighting ?? '', encoding: 'UTF-8');

        $weights = null;

        if($weighting !== '') {
            if(! str_contains($weighting, ':')) {
                throw new SpecificationException('randomisation.arm_weighting must contain a ratio');
            }

            $weights = explode(':', $weighting);

            foreach ($weights as $key => $weight) {
                $weight = mb_trim($weight, encoding: 'UTF-8');
                if($weight === '' || ! ctype_digit($weight)) {
                    throw new SpecificationException(sprintf('randomisation.arm_weighting[%s] must be a positive integer.', $key));
                }
                if($weight < 1) {
                    throw new SpecificationException(sprintf('randomisation.arm_weighting[%s] >= 1', $key));
                }
                $weights[$key] = (int)$weight;
            }
        }

        if(empty($arms)) throw new SpecificationException('randomisation.arms cannot be empty');

        if($weights !== null && count($arms) !== count($weights)) {
            throw new SpecificationException('randomisation.arm_weighting and randomisation.arms must have the same number');
        }

        $armMap = [];

        $i = 0;

        foreach($arms as $key => $arm) {

            if(is_string($arm)) {
                $armName = $arm;
                $armExport = null;
            }
            else {
                $armName = $key;
                $armExport = $arm;
            }

            $armName = mb_trim($armName, encoding: 'UTF-8');

            if ($armName === '') {
                throw new SpecificationException(sprintf('randomisation.arms[%s] arm must not be empty', $i));
            }

            if (isset($armMap[$armName])) {

                throw new SpecificationException(sprintf('randomisation.arms[%s] "%s" is duplicated', $i, $armName));
            }

            $armMap[$armName] = new Arm($armName, $weights[$i] ?? 1, $i, $armExport); //setting all as one means equal split

            $i++;

        }

        $this->arms = $armMap;

        if($this->system === SystemEnum::REDcap || $this->system === SystemEnum::MACRO) {
            if(empty($setVariable = mb_trim($setVariable, encoding: 'UTF-8'))) {
                throw new SpecificationException('randomisation.set_variable is required.');
            }
        }
        else {
            if($setVariable !== null) {
                throw new SpecificationException('randomisation.set_variable must not be set for this system.');
            }
        }

        $this->set_variable = $setVariable;

        if($simpleForFirst !== null) {
            if(!is_int($simpleForFirst)) throw new SpecificationException('safeguarding.simple_for_first must be an integer');
            if($simpleForFirst < 1) throw new SpecificationException('safeguarding.simple_for_first must be a positive number');
        }

        $this->simple_for_first = $simpleForFirst;

        $forceValuesMap = [];

        foreach($forceValues as $key => $forceValue) {
            if (!isset($forceValue['variable'], $forceValue['value'])) {
                throw new SpecificationException(sprintf('force_values[%s] must contain variable and value.', $key));
            }

            $var = mb_trim($forceValue['variable'], encoding: 'UTF-8');

            if ($var === '') {
                throw new SpecificationException(sprintf('force_values[%s] variable must not be empty', $key));
            }

            if (isset($forceValuesMap[$var])) {

                throw new SpecificationException(sprintf('force_values[%s] "%s" is duplicated', $key, $var));
            }

            $forceValuesMap[$var] = new ForceValue($var, $forceValue['value'], $key);
        }

        $this->force_values = $forceValuesMap;

        $this->pid_generate = (bool)($pid['generate'] ?? false);

        if($pid !== [] && $this->system === SystemEnum::Phoenix) {
            throw new SpecificationException('pid configuration must be empty for Phoenix');
        }
        else {
            if($this->pid_generate) {

                if(empty($pid_format = mb_trim($pid['format'] ?? '', encoding: 'UTF-8'))) {
                    throw new SpecificationException('pid.format is required.');
                }
                if(empty($pid_set_variable = mb_trim($pid['set_variable'] ?? '', encoding: 'UTF-8'))) {
                    throw new SpecificationException('pid.set_variable is required.');
                }
                $this->pid_format = $pid_format;
                $this->pid_set_variable = $pid_set_variable;
            }
            else {
                if(isset($pid['format'])) {
                    throw new SpecificationException('pid.format should not be set if pid.generate is false');
                }
                if(isset($pid['set_variable'])) {
                    throw new SpecificationException('pid.set_variable should not be set if pid.generate false');
                }

                $this->pid_format = null;
                $this->pid_set_variable = null;
            }
        }

        $apiToken = mb_trim($apiToken ?? '', encoding: 'UTF-8');

        $this->api_token = empty($apiToken) ? null : $apiToken;

        if($this->system === SystemEnum::Phoenix && $this->api_token !== null) {
            throw new SpecificationException('api_token should not be set for Phoenix');
        }
        else {
            if (empty($this->api_token)) {
                throw new SpecificationException('api_token is required.');
            }
        }

        $requiredVariablesMap = [];

        foreach($requiredVariables as $key => $variable) {
            if(empty($variable = mb_trim($variable, encoding: 'UTF-8'))) {
                throw new SpecificationException(sprintf('required_variables[%s] must not be empty', $key));
            }

            $requiredVariablesMap[] = $variable;
        }

        $this->required_variables = $requiredVariablesMap;

        $actionBy = mb_trim($actionBy ?? '', encoding: 'UTF-8');

        $this->action_by = empty($actionBy) ? null : $actionBy;

        if($this->system === SystemEnum::Phoenix && $this->action_by !== null) {
            throw new SpecificationException('action_by should not be set for Phoenix');
        }
        else {
            if (empty($this->action_by)) {
                throw new SpecificationException('action_by is required.');
            }
        }

    }

    public abstract function getType(): string;

}
