<?php

declare(strict_types=1);

namespace Atlas\RandomisationBundle\Service\Randomisation;

use Atlas\RandomisationBundle\Contract\AlgorithmInterface;
use Atlas\RandomisationBundle\Contract\SpecificationInterface;
use Atlas\RandomisationBundle\Dto\RandomisationResultDto;
use Atlas\RandomisationBundle\Exception\RandomisationException;
use Atlas\RandomisationBundle\Service\Specification\SpecificationLoader;
use DateTimeInterface;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\AutowireLocator;
use Symfony\Component\ExpressionLanguage\ExpressionLanguage;
use Symfony\Component\Uid\Uuid;
use Symfony\Contracts\Service\ServiceProviderInterface;

final readonly class Randomiser
{
    /**
     * @param array $enabled_randomisation
     * @param ServiceProviderInterface $algorithms
     * @param SpecificationLoader $specification_loader
     */
    public function __construct(
        #[Autowire(param: 'shared.randomisation.algorithms')]
        private array $enabled_randomisation,
        #[AutowireLocator('atlas.randomisation.algorithm', indexAttribute: 'algorithm')]
        private ServiceProviderInterface $algorithms,
        private SpecificationLoader $specification_loader,
        private ExpressionLanguage $calculator
    )
    {}

    /**
     * @param string $studyCode
     * @param string $randomisationName
     * @param string $location
     * @param int $version
     * @param string $participantIdentifier
     * @param string $actionBy
     * @param array $variables
     * @param DateTimeInterface|null $simulate
     * @param Uuid|null $simulateId
     * @param SpecificationInterface|null $specification
     * @return RandomisationResultDto
     * @throws InvalidArgumentException
     */
    public function randomise(
        string $studyCode,
        string $randomisationName,
        string $location,
        int $version,
        string $participantIdentifier,
        string $actionBy,
        array $variables = [],
        ?DateTimeInterface $simulate = null,
        ?Uuid $simulateId = null,
        ?SpecificationInterface $specification = null,
    ): RandomisationResultDto
    {
        $specification = $specification ?? $this->specification_loader->load($studyCode, $randomisationName, $version);

        $studyCode = mb_strtoupper($studyCode, encoding: 'utf-8');

        $algorithm = $this->getAlgorithm($specification->getType());

        if($algorithm === null) {
            throw new RandomisationException(sprintf(
                'Cannot find algorithm for type "%s" (code: %s, study: %s, version: %d)',
                $specification->getType(), $randomisationName, $studyCode, $version
            ));
        }

        foreach($specification->required_variables as $variable) {
            if(!array_key_exists($variable, $variables)) {
                throw new RandomisationException(sprintf('Required variable "%s" does not exist', $variable));
            }
        }

        if($specification->when) {
            $when = str_replace(['{', '}'], '', $specification->when);

            if($this->calculator->evaluate($when, $variables)) {
                throw new RandomisationException('Randomisation not ready to open for participant');
            }
        }

        $result = $algorithm->randomise(
            $studyCode,
            $randomisationName,
            $specification,
            $participantIdentifier,
            $location,
            $actionBy,
            $variables,
            simulate: $simulate,
            simulateId: $simulateId
        );

        $result->metadata['randomisation_code'] = mb_strtoupper($specification->code);

        return $result;
    }

    private function getAlgorithm(string $algorithm): ?AlgorithmInterface
    {
        $algorithm = mb_trim(mb_strtolower($algorithm, encoding: 'UTF-8'), encoding: 'UTF-8');

        if($algorithm === '') return null;

        if(! $this->algorithms->has($algorithm)) return null;

        if(! in_array($algorithm, $this->enabled_randomisation, true)) return null;

        return $this->algorithms->get($algorithm);
    }
}
